
/*
  Copyright (C) 2002-2006  Dmitry V. Levin <ldv@altlinux.org>

  The match interface for locate.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*/

#include <stdio.h>
#include <errno.h>
#include <error.h>
#include <string.h>
#include <sys/stat.h>

#include "match.h"
#include "xmalloc.h"

static bool
is_glob(const char *pattern)
{
	const char *p;

	return strchr(pattern, '*') || strchr(pattern, '?')
		|| ((p = strchr(pattern, '[')) && strchr(p + 1, ']'));
}

match_context *
match_init(int argc, char *const *argv, locate_settings *settings)
{
	int     i;
	match_context *ctxv = xcalloc((size_t) argc, sizeof(*ctxv));

	for (i = 0; i < argc; ++i)
	{
		void    (*f) (const char *, match_context *,
			      locate_settings *);

		if (settings->match_regex)
			f = regex_match_init;
		else if (is_glob(argv[i]))
			f = glob_match_init;
		else
			f = substr_match_init;

		f(argv[i], ctxv + i, settings);
	}

	return ctxv;
}

/* Return true if SAMPLE matches to argv.  */
static bool
match_sample(int argc, match_context *ctxv, const char *sample,
	     locate_settings *settings)
{
	int     i;

	for (i = 0; i < argc; ++i)
	{
		if (!ctxv[i].matcher(sample, ctxv + i))
			return false;

		if (!settings->match_all)
			return true;
	}

	return true;
}

/* Return true if SAMPLE matches to argv.  */
bool
match(int argc, match_context *ctxv, const char *path,
      const char *sample, locate_settings *settings)
{
	if (!argc)
		return false;

	if (!match_sample(argc, ctxv, sample, settings))
		return false;

	if (settings->check_existence)
	{
		struct stat st;

		if ((settings->follow_symlinks ? stat : lstat) (path, &st))
			return false;
	}

	if (settings->print_match)
	{
		if (settings->separator == '\n')
			puts(path);
		else
			printf("%s%c", path, settings->separator);
	}
	return true;
}
