
/*
  Copyright (C) 2002-2006  Dmitry V. Levin <ldv@altlinux.org>

  make locate database.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*/

#include <stdio.h>
#include <errno.h>
#include <error.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <sys/stat.h>

#include "xmalloc.h"
#include "lists.h"
#include "updatedb.h"

#define DEFAULT_SEARCH_PATHS "/"
#define DEFAULT_PRUNE_PATHS "/afs /media /mnt /net /proc /tmp /usr/tmp /var/tmp /sys"
#define DEFAULT_PRUNE_FS "afs autofs cardfs cefs cifs coda davfs devfs devpts ftpfs iso9660 localfs mfs ncp ncpfs nfs nfs4 proc shfs smbfs sshfs subfs supermount sysfs tmpfs udf usbdevfs vfat"
#define DEFAULT_DBFILE "/var/lib/locate/locatedb"
#define DEFAULT_MODE 0644
#define DEFAULT_RUNAS "slocate"

#define PACKAGE_BUGREPORT "http://bugs.altlinux.ru/"

static void __attribute__((__noreturn__))
print_usage(const char *str)
{
	if (str)
		error(EXIT_SUCCESS, 0, "%s", str);

	fprintf(stderr, "Try `%s --help' for more information.\n",
		program_invocation_short_name);
	exit(EXIT_FAILURE);
}

static void __attribute__((__noreturn__))
print_help(void)
{
	printf(
"Usage: %s [OPTION]...\n"
"Make a file name database for locate.\n"
"Valid options are:\n"
"--searchpaths='dir1 dir2...'    directories to be put in the database;\n"
"--prunepaths='dir1 dir2...'     unwanted directories;\n"
"--prunefs='fs1 fs2...'          unwanted filesystems;\n"
"--output=dbfile                 database to write output;\n"
"--mode=mode                     database permissions;\n"
"--runas=user                    user to run as;\n"
"--keep-cap                      keep CAP_DAC_READ_SEARCH capability;\n"
"--help                          print this text.\n",
	       program_invocation_short_name);
	printf("\nReport bugs to %s.\n", PACKAGE_BUGREPORT);
	exit(EXIT_SUCCESS);
}

static struct option const longopts[] = {
	{"searchpaths", required_argument, 0, 's'},
	{"prunepaths", required_argument, 0, 'p'},
	{"prunefs", required_argument, 0, 'f'},
	{"output", required_argument, 0, 'o'},
	{"mode", required_argument, 0, 'm'},
	{"runas", required_argument, 0, 'u'},
	{"keep-cap", no_argument, 0, 'k'},
	{"help", no_argument, 0, 'h'},
	{0, no_argument, 0, 0}
};

static const char *
xgetenv(const char *name)
{
	const char *p = getenv(name);

	return (p && *p) ? p : 0;
}

static char *dbtmp = 0;
static pid_t parent_pid;

static void
unlink_dbtmp(void)
{
	if (dbtmp && parent_pid == getpid())
	{
		(void) unlink(dbtmp);
		dbtmp = 0;
	}
}

int
main(int argc, char *argv[])
{
	int     optc;
	const char *runas = 0;
	int     keep_cap = xgetenv("KEEPCAP") ? atoi(xgetenv("KEEPCAP")) : 0;
	const char *dbname = xgetenv("DBFILE") ? : DEFAULT_DBFILE;
	int     fd;
	mode_t  mode = DEFAULT_MODE;

	while ((optc =
		getopt_long(argc, argv, "s:p:f:o:m:u:kh", longopts,
			    (int *) 0)) != -1)
		switch (optc)
		{
			case 's':
				add_search_paths(optarg);
				break;

			case 'p':
				add_prune_path(optarg);
				break;

			case 'f':
				add_prune_fs(optarg);
				break;

			case 'o':
				dbname = xstrdup(optarg);
				break;

			case 'm':
				mode = 0666 & strtol(optarg, 0, 0);
				break;

			case 'u':
				runas = xstrdup(optarg);
				break;

			case 'k':
				keep_cap = 1;
				break;

			case 'h':
				print_help();

			default:
				print_usage(NULL);
		}

	if (optind < argc)
		print_usage("Too many arguments");

	if (!runas && !getuid())
		runas = xgetenv("RUNAS") ? : DEFAULT_RUNAS;

	if (!size_search_paths)
		add_search_paths(xgetenv("SEARCHPATHS") ? :
				 DEFAULT_SEARCH_PATHS);

	if (!size_prune_path)
		add_prune_path(xgetenv("PRUNEPATHS") ? : DEFAULT_PRUNE_PATHS);

	if (!size_prune_fs)
		add_prune_fs(xgetenv("PRUNEFS") ? : DEFAULT_PRUNE_FS);

	convert_fs_paths();

	xasprintf(&dbtmp, "%s.XXXXXX", dbname);
	fd = mkstemp(dbtmp);
	if (fd < 0)
		error(EXIT_FAILURE, errno, "mkstemp: %s", dbtmp);

	parent_pid = getpid();
	atexit(unlink_dbtmp);

	updatedb(fd, runas, keep_cap);

	if (keep_cap)
		mode &= 0660;

	if (fchmod(fd, mode) < 0)
		error(EXIT_FAILURE, errno, "fchmod: %s", dbtmp);

	if (rename(dbtmp, dbname) < 0)
	{
		error(EXIT_FAILURE, errno, "rename: \"%s\" --> \"%s\"",
		      dbtmp, dbname);
	}

	dbtmp = 0;
	return 0;
}
