
/*
  Copyright (C) 2002-2006  Dmitry V. Levin <ldv@altlinux.org>

  The regex method for locate.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*/

#include <stdio.h>
#include <errno.h>
#include <error.h>
#include <string.h>
#include <stdlib.h>
#include <regex.h>

#include "match.h"
#include "xmalloc.h"

static void __attribute__((noreturn))
reg_fatal(int rc, regex_t *preg)
{
	size_t  len = regerror(rc, preg, 0, 0UL);
	char   *errbuf = alloca(len);

	regerror(rc, preg, errbuf, len);
	error(EXIT_FAILURE, 0, "regcomp: %s", errbuf);
	exit(EXIT_FAILURE);
}

static bool
regex_match(const char *sample, match_context *context)
{
	int     rc;
	bool    retval;
	regex_t *preg = context->data;

	rc = regexec(preg, sample, 0UL, 0, 0);
	if (rc == 0)
		retval = true;
	else if (rc == REG_NOMATCH)
		retval = false;
	else
		reg_fatal(rc, preg);

	return retval;
}

void
regex_match_init(const char *pattern, match_context *context,
		  locate_settings *settings)
{
	int     rc;
	regex_t *preg = xcalloc(1UL, sizeof(*preg));

	rc = regcomp(preg, pattern,
		     (settings->ignore_case ? REG_ICASE : 0) | REG_EXTENDED |
		     REG_NEWLINE | REG_NOSUB);
	if (rc)
		reg_fatal(rc, preg);

	context->matcher = regex_match;
	context->data = (void *) preg;
	context->ignore_case = settings->ignore_case;
}
