
/* 
  $Id: frencode.c,v 1.5 2005/12/02 14:33:47 ldv Exp $
  Copyright (C) 1994, 2002 Free Software Foundation, Inc.
  Copyright (C) 2005  Dmitry V. Levin <ldv@altlinux.org>

  front-compress encoding.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*/

/* Uses front compression (also known as incremental encoding);
   see ";login:", March 1983, p. 8.

   The frcode() function gets NUL-terminated string and outputs encoded entity.
   Each entity consists of an offset-differential count byte (the additional
   number of characters of prefix of the preceding entry to use beyond the
   number that the preceding entry is using of its predecessor), followed by a
   null-terminated ASCII remainder.

   If the offset-differential count is larger than can be stored in a byte
   (+/-127), the byte has the value 0x80 and the count follows in a 2-byte
   word, with the high byte first (network byte order).

   Example:

   Input, with NULs changed to newlines:
   /usr/src
   /usr/src/cmd/aardvark.c
   /usr/src/cmd/armadillo.c
   /usr/tmp/zoo

   Length of the longest prefix of the preceding entry to share:
   0 /usr/src
   8 /cmd/aardvark.c
   14 rmadillo.c
   5 tmp/zoo

   Output, with NULs changed to newlines and count bytes made printable:
   0 LOCATE02
   0 /usr/src
   8 /cmd/aardvark.c
   6 rmadillo.c
   -9 tmp/zoo

   (6 = 14 - 8, and -9 = 5 - 14)

   Written by James A. Woods <jwoods@adobe.com>.
   Modified by David MacKenzie <djm@gnu.ai.mit.edu>.
   Modified by Dmitry V. Levin <ldv@altlinux.org>.
*/

#include <stdio.h>
#include <errno.h>
#include <error.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "locatedb.h"
#include "xmalloc.h"
#include "frencode.h"

static void
xwrite(int fd, const char *buffer, size_t count)
{
	ssize_t offset = 0;

	while (count > 0)
	{
		ssize_t block = write(fd, &buffer[offset], count);

		if ((block < 0) && (errno == EINTR))
			continue;
		if (block <= 0)
			error(1, errno, "xwrite");
		offset += block;
		count -= block;
	}
}

/*
 * Return the length of the longest common prefix of strings S1 and S2.
 */

static inline int
prefix_length(const char *s1, const char *s2)
{
	register const char *start;

	for (start = s1; *s1 == *s2 && *s1 != '\0'; ++s1, ++s2)
		;
	return s1 - start;
}

void
frencode(int fd, const char *path)
{
	static const char *old_path;
	static int old_prefix_len;

	int     prefix_len = prefix_length(old_path ? : "", path);
	int     diff = prefix_len - old_prefix_len;

	free((char *) old_path);
	old_path = xstrdup(path);
	old_prefix_len = prefix_len;

	{
		static int need = 1;

		if (need)
		{
			xwrite(fd, LOCATEDB_MAGIC, sizeof(LOCATEDB_MAGIC));
			need = 0;
		}
	}

	/* If the difference is small, it fits in one byte;
	   otherwise, two bytes plus a marker noting that fact.  */
	if (diff < -127 || diff > 127)
	{
		/* XXX diff may overflow short int */
		char    buf[3] = { 0x80, (diff >> 8), diff };

		xwrite(fd, buf, sizeof(buf));
	} else
	{
		char    ch = diff;

		xwrite(fd, &ch, 1UL);
	}

	xwrite(fd, path + prefix_len, strlen(path + prefix_len) + 1);
}
