
/*
  $Id: getstr.c,v 1.5 2005/12/02 14:21:30 ldv Exp $
  Copyright (C) 1993, 1996, 1997, 1998, 2002 Free Software Foundation, Inc.
  Copyright (C) 2005  Dmitry V. Levin <ldv@altlinux.org>

  The getstr implementaion.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*/

/*
  Written by Jan Brittenson, bson@gnu.ai.mit.edu.
  Modified by Dmitry V. Levin <ldv@altlinux.org>
*/

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <error.h>
#include "xmalloc.h"
#include "getstr.h"

#define MIN_SIZE 512

static inline int
inval(void)
{
	errno = EINVAL;
	return -1;
}

/* Read up to (and including) a null from STREAM into *LINEPTR
   + OFFSET (and null-terminate it). *LINEPTR is a pointer returned from
   malloc (or NULL), pointing to *N characters of space.  It is realloc'd
   as necessary.  Return the number of characters read (not including the
   null terminator), or -1 on error or EOF.  */

ssize_t
getstr(char **lineptr, size_t * n, FILE * stream, size_t offset)
{
	ssize_t nchars_avail;	/* Allocated but unused chars in *LINEPTR.  */
	char   *read_pos;	/* Where we're reading into *LINEPTR. */

	if (!lineptr || !n || !stream)
		return inval();

	if (!*lineptr)
	{
		if (offset)
			return inval();
		*n = MIN_SIZE;
		*lineptr = xmalloc(*n);
	} else if (offset > *n)
		return inval();

	nchars_avail = *n - offset;
	read_pos = *lineptr + offset;

	for (;;)
	{
		int     c;

		/* We always want at least one char left in the buffer, since we
		   always (unless we get an error while reading the first char)
		   NUL-terminate the line buffer.  */

		if (nchars_avail < 2)
		{
			if (*n >= MIN_SIZE)
				*n <<= 1;
			else
				*n = MIN_SIZE;

			nchars_avail = *lineptr + *n - read_pos;
			*lineptr = xrealloc(*lineptr, *n);
			read_pos = *lineptr + *n - nchars_avail;
		}

		c = getc(stream);
		if (c == EOF || ferror(stream))
		{
			/* Return partial line, if any.  */
			if (read_pos == *lineptr)
				return -1;
			else
				break;
		}

		*read_pos++ = c;
		--nchars_avail;

		if (!c)
			break;
	}

	/* Done - NUL terminate and return the number of chars read.  */
	*read_pos = '\0';

	return read_pos - (*lineptr + offset);
}
