
/*
  $Id: locate.c,v 1.8 2005/12/02 14:33:47 ldv Exp $
  Copyright (C) 1994 Free Software Foundation, Inc.
  Copyright (C) 2002, 2004, 2005  Dmitry V. Levin <ldv@altlinux.org>

  locate -- search databases for filenames that match patterns

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA

  Usage: locate [options] pattern...

  Scan a pathname list for the full pathname of a file, given only
  a piece of the name (possibly containing shell globbing metacharacters).
  The list has been processed with front-compression, which reduces
  the list size by a factor of 4-5.

  Described more fully in Usenix ;login:, Vol 8, No 1,
  February/March, 1983, p. 8.

  Written by James A. Woods <jwoods@adobe.com>.
  Modified by David MacKenzie <djm@gnu.ai.mit.edu>.
  Modified by Dmitry V. Levin <ldv@altlinux.org>.
*/

#include <stdio.h>
#include <errno.h>
#include <error.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <fnmatch.h>
#include <regex.h>
#include <getopt.h>
#include <locale.h>
#include <sys/stat.h>

#include "nextelem.h"
#include "methods.h"
#include "xmalloc.h"
#include "locatedb.h"

extern int __libc_enable_secure;

/* Warn if a database is older than this.  8 days allows for a weekly
   update that takes up to a day to perform.  */
#define WARN_NUMBER_UNITS (8)

#define SECONDS_PER_UNIT (60 * 60 * 24)

#define WARN_SECONDS ((SECONDS_PER_UNIT) * (WARN_NUMBER_UNITS))

/* Printable name of units used in WARN_SECONDS */
static const char warn_name_units[] = "days";

/*
 * Open DBFILE and call corresponding locate METHOD.
 * Return number of lines matched.
 */
static unsigned
locate(const char *pattern, const char *dbfile, match_context * ctx,
       int icase, int echeck)
{
	/* The pathname database.  */
	FILE   *fp;

	/* Number of bytes read from an entry.  */
	int     nread;

	/* The return value.  */
	unsigned rc;

	/* To check the age of the database.  */
	struct stat st;
	time_t  now;

	char    path[sizeof(LOCATEDB_MAGIC)];

	if (!(fp = fopen(dbfile, "r")) || fstat(fileno(fp), &st))
	{
		error(EXIT_SUCCESS, errno, "%s", dbfile);
		return 0;
	}
	time(&now);
	if (now - st.st_mtime > WARN_SECONDS)
	{
		/* For example:
		   warning: database `fred' is more than 8 days old */
		error(EXIT_SUCCESS, 0,
		      "warning: database `%s' is more than %d %s old",
		      dbfile, WARN_NUMBER_UNITS, warn_name_units);
	}

	nread = fread(path, 1UL, sizeof(LOCATEDB_MAGIC), fp);
	if (nread != sizeof(LOCATEDB_MAGIC)
	    || memcmp(path, LOCATEDB_MAGIC, sizeof(LOCATEDB_MAGIC)))
	{
		error(EXIT_SUCCESS, 0,
		      "warning: file `%s' is not a locate database", dbfile);
		return 0;
	}

	rc = match(ctx, pattern, fp, dbfile, icase, echeck);

	if (fclose(fp) == EOF)
		error(EXIT_SUCCESS, errno, "%s", dbfile);

	return rc;
}

static char *
xgetenv(const char *name)
{
	char   *p = getenv(name);

	return (p && *p) ? p : 0;
}

static void
drop_priv(void)
{
	gid_t   gid = getgid();

	if (setresgid(gid, gid, gid))
		error(EXIT_FAILURE, errno, "setresgid: %u", gid);

	__libc_enable_secure = 0;
}

static void __attribute__ ((__noreturn__)) usage(FILE * stream, int status)
{
	fprintf(stream, "\
Usage: %s [-d path | --database=path] [-e | --existing]\n\
      [-i | --ignore-case] [-r | --regexp] [--help] pattern...\n", program_invocation_short_name);
	fputs("\nReport bugs to http://bugs.altlinux.ru/\n", stream);
	exit(status);
}

static struct option const longopts[] = {
	{"database", required_argument, 0, 'd'},
	{"existing", no_argument, 0, 'e'},
	{"no-existing", no_argument, 0, 'E'},
	{"ignore-case", no_argument, 0, 'i'},
	{"regexp", no_argument, 0, 'r'},
	{"help", no_argument, 0, 'h'},
	{0, no_argument, 0, 0}
};

static int
is_glob(const char *pattern)
{
	const char *p;

	return strchr(pattern, '*') || strchr(pattern, '?')
		|| ((p = strchr(pattern, '[')) && strchr(p + 1, ']'));
}

int
main(int argc, char **argv)
{
	const char *dbpath = LOCATEDB_DBFILE;
	int     optc;
	int     icase = 0;
	int     match_regexp = 0;
	int     echeck = 0, echeck_set = 0;
	unsigned found = 0;

	setlocale(LC_ALL, "");

	if (xgetenv("LOCATE_PATH"))
	{
		drop_priv();
		dbpath = xgetenv("LOCATE_PATH");
	}

	while ((optc =
		getopt_long(argc, argv, "d:eEhir", longopts,
			    (int *) 0)) != -1)
		switch (optc)
		{
			case 'd':
				drop_priv();
				dbpath = optarg;
				break;

			case 'e':
				echeck = 1;
				echeck_set = 1;
				break;

			case 'E':
				echeck = 0;
				echeck_set = 1;
				break;

			case 'i':
				icase = 1;
				break;

			case 'r':
				match_regexp = 1;
				break;

			case 'h':
				usage(stdout, EXIT_SUCCESS);

			default:
				usage(stderr, EXIT_FAILURE);
		}

	if (optind == argc)
		usage(stderr, 1);

	if (!echeck_set)
		echeck = __libc_enable_secure;

	if (__libc_enable_secure && !echeck)
		drop_priv();

	for (; optind < argc; ++optind)
	{
		const char *e;

		next_element(dbpath);
		while ((e = next_element(0)))
		{
			match_context ctx;

			memset(&ctx, 0, sizeof(ctx));
			if (match_regexp)
			{
				ctx.open = regex_open;
				ctx.close = regex_close;
				ctx.match = regex_match;
			} else if (is_glob(argv[optind]))
			{
				ctx.open = glob_open;
				ctx.match = glob_match;
			} else
			{
				ctx.open = substr_open;
				ctx.match = substr_match;
			}

			found |= locate(argv[optind], e, &ctx, icase, echeck);
		}
	}

	return !found;
}
