
/*
  $Id: match.c,v 1.6 2005/12/02 14:21:30 ldv Exp $
  Copyright (C) 2002, 2003, 2005  Dmitry V. Levin <ldv@altlinux.org>

  The common match interface for locate.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*/

#include <stdio.h>
#include <errno.h>
#include <error.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>

#include "methods.h"
#include "getstr.h"
#include "xmalloc.h"

/* Print the entries in FP that match pattern PATTERN.
   Return the number of entries printed.  */
unsigned
match(match_context * ctx, const char *pattern, FILE * fp, const char *dbfile,
      int icase, int echeck)
{
	/* The return value.  */
	unsigned printed = 0;

	/* An input byte.  */
	int     c;

	/* Number of bytes read from an entry.  */
	int     nread = 0;

	/* The current input database entry.  */
	char   *path;

	/* Amount allocated for it.  */
	size_t  pathsize;

	/* The length of the prefix shared with the previous database entry.  */
	int     count = 0;

	/* The handle used by locate methods.  */
	void   *handle = ctx->open(pattern, icase);

	pathsize = 1026;	/* Increased as necessary by getstr.  */
	path = xmalloc(pathsize);

	c = fgetc(fp);
	while (c != EOF)
	{
		if (c == 0x80)
		{
			short   c1, c2, r;

			if (((c1 = fgetc(fp)) == EOF)
			    || ((c2 = fgetc(fp)) == EOF))
			{
				error(EXIT_SUCCESS, errno,
				      "%s: unexpected end of input", dbfile);
				break;
			}

			r = (c1 << 8) | (c2 & 0xff);
			count += r;
		} else if (c > 127)
			count += c - 256;
		else
			count += c;

		if (nread && (count > nread))
		{
			error(EXIT_SUCCESS, 0,
			      "%s: corrupted database", dbfile);
			break;
		}

		/* Overlay the old path with the remainder of the new.  */
		nread = getstr(&path, &pathsize, fp, count);
		if (nread < 0)
		{
			if (EINVAL == errno)
				error(EXIT_SUCCESS, 0,
				      "%s: corrupted database", dbfile);
			break;
		}
		nread += count;
		c = fgetc(fp);

		if (ctx->match(handle, path, icase))
		{
			struct stat st;

			if (!echeck || !lstat(path, &st))
			{
				puts(path);
				++printed;
			}
		}
	}

	if (ferror(fp))
		error(EXIT_SUCCESS, errno, "%s", dbfile);

	if (ctx->close)
		ctx->close(handle);

	return printed;
}
