
/*
  $Id: scantree.c,v 1.9 2005/12/02 14:33:47 ldv Exp $
  Copyright (C) 2002-2005  Dmitry V. Levin <ldv@altlinux.org>

  The scantree implementation.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*/

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <error.h>
#include <unistd.h>
#include <fts.h>
#include <sys/vfs.h>

#include "lists.h"
#include "scantree.h"
#include "frencode.h"
#include "xmalloc.h"

static int
ftsent_compare(const FTSENT ** a, const FTSENT ** b)
{
	return strcmp((*a)->fts_name, (*b)->fts_name);
}

static int
string_compare(const char *a, const char **b)
{
	size_t  len = strlen(*b);

	return strncmp(a, *b, len) ? : (a[len] != '\0');
}

static int
long_compare(const long *a, const long *b)
{
	return a - b;
}

static int
is_excluded(const char *path)
{
	struct statfs stb;
	static long fstype;
	static int rc;

	/* first check pruned paths */
	if (bsearch(path, list_prune_path, size_prune_path,
		    sizeof(*list_prune_path),
		    (comparison_fn_t) string_compare))
		return 1;

	/* then pruned fs types */
	if (statfs(path, &stb))
		return 0;

	/* cached? */
	if (stb.f_type == fstype)
		return rc;
	fstype = stb.f_type;

	rc = bsearch(&fstype, list_prune_fstype, fstype_list_size,
		     sizeof(*list_prune_fstype),
		     (comparison_fn_t) long_compare) ? 1UL : 0UL;

	return rc;
}

void
scantree(int fd)
{
	FTS    *handle;
	FTSENT *ent;

	list_search_paths = (const char **) xrealloc(list_search_paths,
						     ++size_search_paths *
						     sizeof
						     (*list_search_paths));
	list_search_paths[size_search_paths - 1] = '\0';

	handle = fts_open((char *const *) list_search_paths,
			  FTS_PHYSICAL | FTS_NOSTAT, ftsent_compare);
	if (!handle)
		error(EXIT_FAILURE, errno, "fts_open");

	while ((ent = fts_read(handle)))
		switch (ent->fts_info)
		{
			case FTS_D:
				if (is_excluded(ent->fts_path))
					fts_set(handle, ent, FTS_SKIP);
				else
					frencode(fd, ent->fts_path);
				break;
			case FTS_DC:
			case FTS_DP:
			case FTS_ERR:
			case FTS_NS:
				break;
			default:
				frencode(fd, ent->fts_path);
		}

	if (errno)
		error(EXIT_FAILURE, errno, "fts_read");

	if (fts_close(handle))
		error(EXIT_FAILURE, errno, "fts_close");
}
