
/*
  $Id: updatedb.c,v 1.5 2005/12/02 14:33:47 ldv Exp $
  Copyright (C) 2002, 2005  Dmitry V. Levin <ldv@altlinux.org>

  The updatedb implementation.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*/

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <error.h>
#include <unistd.h>
#include <sys/wait.h>
#include <pwd.h>
#include <grp.h>
#include <sys/capability.h>
#include <sys/prctl.h>

#include "updatedb.h"
#include "scantree.h"

void
updatedb(int fd, const char *runas, int keep_cap)
{
	struct passwd *pw;
	pid_t   pid;

	if (!runas)
	{
		scantree(fd);
		return;
	}

	if (!(pw = getpwnam(runas)))
		error(EXIT_FAILURE, 0, "lookup for user \"%s\" failed",
		      runas);
	endpwent();

	pid = fork();
	if (pid < 0)
		error(EXIT_FAILURE, errno, "fork");
	if (pid)
	{
		/* parent */
		int     status;

		if (waitpid(pid, &status, 0) < 0)
			error(EXIT_FAILURE, errno, "waitpid");

		if (WIFEXITED(status))
		{
			if (WEXITSTATUS(status))
				error(WEXITSTATUS(status), 0,
				      "child exited abnormally");

			if (keep_cap && fchown(fd, (uid_t) - 1, pw->pw_gid))
				error(EXIT_FAILURE, errno, "fchgrp: %u",
				      pw->pw_gid);

			return;
		} else if (WIFSIGNALED(status))
			error(128 + WTERMSIG(status), 0,
			      "child terminated abnormally");
		else
			error(EXIT_FAILURE, 0,
			      "unrecognized exit status from child");
	}

	if (setgroups(0UL, NULL) < 0)
		error(EXIT_FAILURE, errno, "setgroups");

	if (setgid(pw->pw_gid))
		error(EXIT_FAILURE, errno, "setgid: %u", pw->pw_gid);

	/* child */
	if (keep_cap)
	{
		cap_t   caps;

		if (prctl(PR_SET_KEEPCAPS, 1))
			error(EXIT_FAILURE, errno, "prctl");

		caps = cap_from_text("cap_setuid,cap_dac_read_search=ep");
		if (!caps)
			error(EXIT_FAILURE, errno, "cap_from_text failed");

		if (cap_set_proc(caps) < 0)
			error(EXIT_FAILURE, errno, "cap_set_proc failed");

		cap_free(caps);

		if (setreuid(pw->pw_uid, pw->pw_uid))
			error(EXIT_FAILURE, errno, "setreuid: %u",
			      pw->pw_uid);

		caps = cap_from_text("cap_dac_read_search=ep");
		if (!caps)
			error(EXIT_FAILURE, errno, "cap_from_text failed");

		if (cap_set_proc(caps) < 0)
			error(EXIT_FAILURE, errno, "cap_set_proc failed");

		cap_free(caps);
	} else if (setuid(pw->pw_uid))
		error(EXIT_FAILURE, errno, "setuid: %u", pw->pw_uid);

	scantree(fd);

	exit(EXIT_SUCCESS);
}
