//==========================================================================
//
//      iso9660fs.c
//
//      RedBoot support for iso9660 filesystem
//
//==========================================================================
//####ECOSGPLCOPYRIGHTBEGIN####
// -------------------------------------------
// This file is part of eCos, the Embedded Configurable Operating System.
// Copyright (C) 1998, 1999, 2000, 2001, 2002 Red Hat, Inc.
// Copyright (C) 2003 Gary Thomas <gary@mind.be>
//
// eCos is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2 or (at your option) any later version.
//
// eCos is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
// for more details.
//
// You should have received a copy of the GNU General Public License along
// with eCos; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. However the source code for this file must still be made available
// in accordance with section (3) of the GNU General Public License.
//
// This exception does not invalidate any other reasons why a work based on
// this file might be covered by the GNU General Public License.
//
// Alternative licenses for eCos may be arranged by contacting Red Hat, Inc.
// at http://sources.redhat.com/ecos/ecos-license/
// -------------------------------------------
//####ECOSGPLCOPYRIGHTEND####
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    STC Elins
// Contributors: STC Elins
// Date:         2007-11-29
// Purpose:      
// Description:  
//              
// This code is part of RedBoot (tm).
//
//####DESCRIPTIONEND####
//
//==========================================================================

#include <redboot.h>
#include <fs/disk.h>
#include <fs/ide.h>
#include <fs/iso9660fs.h>
#include <ctype.h>
#define DEBUG_ISO9660FS 0
static void *iso9660fs_open(partition_t *p, const char *path);
static int  iso9660fs_read(void *fp, char *buf, unsigned int nbytes);

// This structure is the only thing exported by this module.
// These filesystem function pointers are attached to disk
// partitions in the generic disk handling code.
//
fs_funs_t redboot_iso9660fs_funs = {
    iso9660fs_open,
    iso9660fs_read
};
struct iso9660_read_info {
    partition_t  *part;     		/* partition holding this filesystem	*/
	unsigned int sectror_start, sector_start_offs;
    unsigned int fflags;
    unsigned int fsize;
    unsigned int fpos;
};
// A single block buffer to be shared carefully.
static struct iso9660_read_info fread_info;
const char *_DISK_READ_ERROR={"disk read error.\n"};
const char *_DISK_NOT_FOUND={" - [disk not found]\n"};
/*============================================================================================================*/
#if (CYG_BYTEORDER == CYG_MSBFIRST)
static unsigned int get_le32_unalligned(unsigned char *a)
{int i;
 unsigned int v;
 for(i=3, v=0; i>=0; i--)
	{v<<=8;
     v|=a[i];
	}
 return v;
}
static unsigned short get_le16_unalligned(unsigned char *a)
{int i;
 unsigned short v;
 for(i=1, v=0; i>=0; i--)
	{v<<=8;
     v|=a[i];
	}
 return v;
}
#else
static unsigned int get_le32_unalligned(unsigned char *a)
{int i;
 unsigned int v;
 for(i=0, v=0; i < 4; i++)
	{v<<=8;
     v|=a[i];
	}
 return v;
}
static unsigned short get_le16_unalligned(unsigned char *a)
{int i;
 unsigned short v;
 for(i=0, v=0; i < 2; i++)
	{v<<=8;
     v|=a[i];
	}
 return v;
}
#endif
/*============================================================================================================*/
#if DEBUG_ISO9660FS > 2 
void do_isofs_dump_dir(struct iso_directory_record *dir_entry)
{unsigned int i;
 char s[256], *p;
 struct iso_directory_record *dir_entry_max;
 unsigned int first_data_zone;
 if(dir_entry == NULL)
	{diag_printf("%s: error.\n", __FUNCTION__);return;}
 dir_entry_max= (struct iso_directory_record *) (((char *)dir_entry) +2048);
 p= (char *)dir_entry;
 for(i=0; dir_entry < dir_entry_max; i++)
	{/* If the length byte is zero, we should move on to the next CDROM sector. If we are at the end of the directory, we kick out of the while loop. */
	 if(*dir_entry->length == 0)
		{diag_printf("%s: dir_entry conmtinue at next cdrom sector\n", __FUNCTION__); return; }
	 diag_printf("%s: directory record %d:(0x%04x)\n", __FUNCTION__, i, ((unsigned int)(((char *)dir_entry) -p)) & 0x00fff);
 	 first_data_zone = get_le32_unalligned(dir_entry->extent) + *dir_entry->ext_attr_length;
	 diag_printf("\tfirst_data_zone= 0x%08x; offs= 0x%02x; flags=0x%02x\n;", first_data_zone, *dir_entry->ext_attr_length, *dir_entry->flags);
	 diag_printf("\tdir_length= %d bytes; file_section_length= 0x%08x;\n",  *dir_entry->length, get_le32_unalligned(dir_entry->size));
	 memcpy(s, dir_entry->name, *dir_entry->name_len);
	 s[*dir_entry->name_len]='\0';
	 diag_printf("\tname_len= %d bytes; name= [%s];\n",  *dir_entry->name_len, s);	
	 dir_entry=(struct iso_directory_record *) (((unsigned char *)dir_entry)+ *dir_entry->length);
	}
 return;
}
#endif
/*============================================================================================================*/
/* Mount an iso9660 filesystem. Return 0 if successful.*/
struct iso_directory_record * iso9660fs_mount(partition_t *part, unsigned int *buf)
{int joliet=0;
 struct iso_primary_descriptor *iso_toc;
 struct iso_supplementary_descriptor *iso_toc2;
 struct iso_directory_record * rootp;
 struct disk *d;
 unsigned int logical_block_size, volume_space_size, first_data_zone;
 d= part->disk;
 iso_toc= (struct iso_primary_descriptor *) buf;
 iso_toc2=(struct iso_supplementary_descriptor *) buf;
 if (!DISK_READ(d,  16 * SECTORS_PER_CDROM_SECTOR, buf, SECTORS_PER_CDROM_SECTOR))
	{diag_printf("%s", _DISK_NOT_FOUND); return NULL;}
 volume_space_size=  get_le32_unalligned(iso_toc->volume_space_size);
 logical_block_size= get_le16_unalligned(iso_toc->logical_block_size);
 if (memcmp (iso_toc->id, "CD001", 5) == 0)
	{switch(*iso_toc->type)
		{case 1:  break;
   		 case 2: if (iso_toc2->escape[0] == 0x25 && iso_toc2->escape[1] == 0x2f) 
					{switch(iso_toc2->escape[2])
						{case 0x40: joliet = 1; break;
				 		 case 0x43: joliet = 2; break;
		    			 case 0x45:	joliet = 3; break;
		    			}
					}
		 default: diag_printf("%s: [%d] error.\n", __FUNCTION__, __LINE__); return NULL;
		}
	}
 rootp = (struct iso_directory_record *) iso_toc->root_directory_record;
 first_data_zone = get_le32_unalligned(rootp->extent);
#if DEBUG_ISO9660FS > 1 
 {char s[256];
  memcpy(s,(char *)(&iso_toc->volume_id), 31); s[31]='\0';
  diag_printf("%s: disk_label=%s; block_size= %d; total_space_size= %d blocks;\n", __FUNCTION__, s, logical_block_size, volume_space_size);
  diag_printf("%s: directory record:\n", __FUNCTION__);
  diag_printf("\tfirst_data_zone= 0x%08x; flags=0x%02x\n", first_data_zone, *rootp->flags);
  diag_printf("\tfirst_data_zone= 0x%08x; offs= 0x%02x; flags=0x%02x\n", first_data_zone, *rootp->ext_attr_length, *rootp->flags);
  diag_printf("\tdir_length= %d bytes; file_section_length= 0x%08x\n;",  *rootp->length, get_le32_unalligned(rootp->size));
  memcpy(s, rootp->name, *rootp->name_len);
  s[*rootp->name_len]='\0';
  diag_printf("\tname_len= %d bytes; name= [%s];\n",  *rootp->name_len, s);
  fread_info.fsize= get_le32_unalligned(rootp->size);
  fread_info.sectror_start= first_data_zone;
 }
#endif
 if (!DISK_READ(d,  first_data_zone* SECTORS_PER_CDROM_SECTOR, buf, SECTORS_PER_CDROM_SECTOR))
	{diag_printf("%s", _DISK_READ_ERROR); return NULL;}
// dump_buf((unsigned char *)buf);
#if DEBUG_ISO9660FS > 2 
 do_isofs_dump_dir((struct iso_directory_record *)buf);
#endif
 return (struct iso_directory_record *)buf;
}
/*============================================================================================================*/
static int iso9660fs_dir_lookup(struct disk *d, struct iso_directory_record *iso9660_dir, struct iso9660_read_info *f_read_info, const char *pathname, unsigned int len)
{unsigned int i, j;
 struct iso_directory_record *dir_entry_max, *dir_entry;
 unsigned int sector_num, sector_start;
 if(iso9660_dir == NULL)
	{diag_printf("%s: error.\n", __FUNCTION__);return 0;}
 dir_entry_max= (struct iso_directory_record *) (((char *)iso9660_dir) +2048);
 sector_num= f_read_info->sectror_start;
 for(i=0, dir_entry= iso9660_dir; dir_entry < dir_entry_max; i++, dir_entry=(struct iso_directory_record *) (((unsigned char *)dir_entry)+ *dir_entry->length)  )
	{if(*dir_entry->length == 0)					
		{f_read_info->fsize-= CDROM_SECTOR_SIZE;
		 if(!f_read_info->fsize)				
			{ return 0;}
		 sector_num++;
 	 	 if (!DISK_READ(d, (sector_num * SECTORS_PER_CDROM_SECTOR), (unsigned int *)iso9660_dir, SECTORS_PER_CDROM_SECTOR))	/* read next_dir_entry sector */
			{diag_printf("%s", _DISK_READ_ERROR); return 0;}
			 dir_entry= iso9660_dir;
		}
 	 sector_start = get_le32_unalligned(dir_entry->extent);
	 if(strchr(pathname, ';')!=NULL)
	 	{if(*dir_entry->name_len != len)
			{continue;}
		}
 	 {unsigned char *p;
	  p= dir_entry->name;
	  for(p= dir_entry->name, j=0; j < len; j++, p++)
		{*p= _tolower(*p);}
	 }
	 if(memcmp(dir_entry->name, pathname, len))
		{continue;}
	 f_read_info->sectror_start= sector_start;
	 f_read_info->sector_start_offs= *dir_entry->ext_attr_length;
	 f_read_info->fflags= *dir_entry->flags;
     f_read_info->fsize= get_le32_unalligned(dir_entry->size);
     f_read_info->fpos= 0;
#if DEBUG_ISO9660FS > 2 
 	 diag_printf("%s: file found at sector 0x%04x\n", __FUNCTION__, sector_start);
#endif
	 return sector_start;
	}
 return 0;
}
/*============================================================================================================*/
// Starting from the given directory, find the inode number, filetype, and parent inode for the given file pathname.
// If successful, fills out ino_info_t and return true.
static int iso9660fs_inode_lookup(struct disk *d, struct iso_directory_record * root_dir, struct iso9660_read_info *f_read_info, const char *pathname)
{int next_dir_sector;
 const char *p, *pe;
 struct iso_directory_record * dir_entry;
 dir_entry= root_dir;
 pe = pathname + strlen(pathname);
 for (p = pathname; p < pe; p++, dir_entry= root_dir) 			
	{if ((*p) != '/') continue;																			/* find next delimiter in path.*/
	 if((p- pathname) <= 1) 
		{pathname= p+1; continue;}
	 if((next_dir_sector = iso9660fs_dir_lookup(d, dir_entry, f_read_info, pathname, p- pathname))==0)		/* find desired subpath in cur_dir (dir_entry) */
		{return 0;}																						/* subpath not fount in cur_dir */
	 if(!(f_read_info->fflags & ISO9660_DIR_FLAG))
		{return 0;}																						/* is not dir (is a file) */
 	 if (!DISK_READ(d, (next_dir_sector * SECTORS_PER_CDROM_SECTOR), (unsigned int *)root_dir, SECTORS_PER_CDROM_SECTOR))	/* read next_dir_entry sector */
		{diag_printf("%s", _DISK_READ_ERROR); return 0;}
	 pathname= p+1;
	 continue;
   	}
 if(pe <= pathname)																						/* dir_ensry is a dir, where file to look forward */
	{diag_printf("incorrect fileneme.\n");return 0;}
 if((next_dir_sector = iso9660fs_dir_lookup(d, dir_entry, f_read_info, pathname, pe- pathname))==0)		/* lookup for file */
	{return 0;}
 if(f_read_info->fflags & ISO9660_DIR_FLAG)																/* is a dir (not a file) */
	{diag_printf("is a directory.\n");return 0;}
 return 1;
}

static void *iso9660fs_open(partition_t *p, const char *filepath)
{unsigned int  buf[CDROM_SECTOR_SIZE/sizeof(unsigned int)];
 struct iso_directory_record *root_dir;
 memset(&fread_info, 0, sizeof(struct iso9660_read_info));
 fread_info.part= p;
 if ((root_dir=iso9660fs_mount(p, buf)) == NULL)  														/* mount cdrom */
	{diag_printf("cdrom mount error.\n");return NULL; }
 if ((iso9660fs_inode_lookup(p->disk, root_dir, &fread_info, filepath)) == 0) 									 /* find file inode */
	{diag_printf("file not found.\n");return NULL;}
 return &fread_info;
}

static int iso9660fs_read(void *fp, char *buf, unsigned int nbytes)
{unsigned int  tbuf[CDROM_SECTOR_SIZE/sizeof(unsigned int)];
 unsigned int desired_cdrom_sector, n, nread;
 struct iso9660_read_info *f_read_stat;
 struct disk *d;
 char *p, *pr;
 f_read_stat= (struct iso9660_read_info *) fp;
 d= f_read_stat->part->disk;
 for(nread= 0, p= buf; nbytes > 0; )
 	{if ((f_read_stat->fpos + nbytes) > f_read_stat->fsize)
 	 	{nbytes = f_read_stat->fsize - f_read_stat->fpos;}
 	 desired_cdrom_sector= f_read_stat->sectror_start + (f_read_stat->fpos/CDROM_SECTOR_SIZE);		/* cdrom swctor to read 			*/
 	 if (!DISK_READ(d, desired_cdrom_sector * SECTORS_PER_CDROM_SECTOR, tbuf, SECTORS_PER_CDROM_SECTOR))	/* read 1 cdrom sector (2048 bytes) */
		{diag_printf("%s", _DISK_READ_ERROR); return 0;}
	 n= f_read_stat->fpos % CDROM_SECTOR_SIZE;
	 pr= ((char *)tbuf) + n;
	 n= CDROM_SECTOR_SIZE- n;
	 if(n > nbytes) n= nbytes;
	 memcpy(p, pr, n);
	 p+= n;
	 f_read_stat->fpos+=n;
	 nread+= n;
	 nbytes-=n;
	}
 return nread;
}
