//==========================================================================
//
//      usb_ohci.c
//
//      RedBoot USB OHCI support
//
//==========================================================================
//####ECOSGPLCOPYRIGHTBEGIN####
// -------------------------------------------
// This file is part of eCos, the Embedded Configurable Operating System.
// Copyright (C) 1998, 1999, 2000, 2001, 2002 Red Hat, Inc.
//
// eCos is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2 or (at your option) any later version.
//
// eCos is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
// for more details.
//
// You should have received a copy of the GNU General Public License along
// with eCos; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. However the source code for this file must still be made available
// in accordance with section (3) of the GNU General Public License.
//
// This exception does not invalidate any other reasons why a work based on
// this file might be covered by the GNU General Public License.
//
// Alternative licenses for eCos may be arranged by contacting Red Hat, Inc.
// at http://sources.redhat.com/ecos/ecos-license/
// -------------------------------------------
//####ECOSGPLCOPYRIGHTEND####
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    STC Elins
// Contributors: STC Elins
// Date:         2007-11-29
// Purpose:      
// Description:  
//              
// This code is part of RedBoot (tm).
//
//####DESCRIPTIONEND####
//
//==========================================================================
#include <redboot.h>
#include <fs/usb_ohci.h>
#include <fs/usb.h>
#include <errno.h>
#include <cyg/hal/hal_intr.h>           
#include <cyg/io/pci_hw.h>
#include <cyg/io/pci.h>
#undef  OHCI_DEBUG
static unsigned int   usb_heap_size;
static unsigned char *usb_heap_base;
static unsigned char *usb_heap_free;
static unsigned int   ohci_found= 0;
static struct ohci_hc ohci_array[USB_MAX_OHCI];
static struct usb_device usb_dev[USB_MAXDEV];
static int usb_dev_found;
static struct ed *ed_0;
unsigned int *usb_buf_0, *usb_buf_1, *usb_buf_2;
extern void mdelay(unsigned int ms);
const char *_WOK={"- ok.\n"};
/*============================================================================*/
static inline void ohci_writel(unsigned int v, volatile unsigned int * addr)
{v= cpu_to_le32(v);
 *addr=v;
 v=*addr;
return;
}
/*============================================================================*/
static inline unsigned int ohci_readl(volatile unsigned int * addr)
{unsigned int v;
 v=*addr;
 v= le32_to_cpu(v);
 return v;
}
/*============================================================================*/
static inline void ohci_disable(struct ohci_hc *ohci)
{ohci->enabled=0; return;}

static inline unsigned int roothub_a (struct ohci_hc *hc)
{unsigned int v = ohci_readl (&hc->regs->roothub.a);
 if (v == -1) ohci_disable (hc);
 return v; 
}
static inline unsigned int roothub_b (struct ohci_hc *hc)
{return ohci_readl (&hc->regs->roothub.b); }
static inline unsigned int roothub_status (struct ohci_hc *hc)
{return ohci_readl (&hc->regs->roothub.status); }
static inline unsigned int roothub_portstatus (struct ohci_hc *hc, int i)
{unsigned int v = ohci_readl (&hc->regs->roothub.portstatus [i]);
 if (v == -1) ohci_disable (hc);
 return v; 
}

#if 0
static char *ohci_regs[25]=
	{"revision",
	 "control",
	 "cmdstatus",
	 "intrstatus",
	 "intrenable",
	 "intrdisable",
	 "hcca",
	 "ed_periodcurrent",
	 "ed_controlhead",
	 "ed_controlcurrent",
	 "ed_bulkhead",
	 "ed_bulkcurrent",
	 "donehead",
	 "fminterval",
	 "fmremaining",
	 "fmnumber",
	 "periodicstart",
	 "lsthresh",
	 "roothub_a",
	 "roothub_b",
	 "roothub_status",
	 "roothub_port_0_status",
	 "roothub_port_1_status",
	 "roothub_port_2_status",
	 "roothub_port_3_status"};
static void ohci_dump_regs(struct ohci_hc *ohci)
{unsigned int i;
 volatile unsigned int *addr, v;
 addr= &ohci->regs->control;
 for(i=1; i < 24; i++, addr++)
	{v= *addr;
	 v=le32_to_cpu(v);
	 diag_printf("\t%-20s %08x\n", ohci_regs[i], v);
	}
 v= le32_to_cpu(ohci->hcca->done_head);
 diag_printf("\t%-20s %08x\n", "done_head", v);
 return;
}
#endif
//#ifdef OHCI_DEBUG
/*=====================================================================================================*/
void ohci_dump_td(struct td *td, char *s)
{diag_printf("%s> td_dma=0x%08x; hwINFO=0x%08x; hwCBP=0x%08x; hwNextTD=0x%08x; hwBE=0x%08x;\n", s, td->td_dma, le32_to_cpu(td->hwINFO), le32_to_cpu(td->hwCBP),le32_to_cpu(td->hwNextTD),le32_to_cpu(td->hwBE));
 return;
}
void ohci_dump__ed(struct ed *ed, const char *s)
{struct td *td;
 unsigned int addr, i;
 char buf[32];
 diag_printf("%s> ed_dma=0x%08x; hwINFO=0x%08x; hwTailP=0x%08x; hwHeadP=0x%08x; hwNextED=0x%08x;\n", s,  ed->ed_dma, le32_to_cpu(ed->hwINFO), le32_to_cpu(ed->hwTailP),le32_to_cpu(ed->hwHeadP),le32_to_cpu(ed->hwNextED));
 addr= le32_to_cpu(ed->hwHeadP);
 for (i=0; i< 10; i++)
	{td= (struct td *) ((((unsigned int) addr) | 0xa0000000U) & 0xfffffff0U);
	 diag_sprintf(buf, "td_%d", i);
	 ohci_dump_td(td, buf);
	 addr= le32_to_cpu(td->hwNextTD);
	 if(le32_to_cpu(td->hwNextTD) ==  td->td_dma) break;
	}
 diag_printf("===================\n");
 return;
}
void ohci_dump_ed(struct ed *ed, const char *s)
{diag_printf("%s> ed_dma=0x%08x; hwINFO=0x%08x; hwTailP=0x%08x; hwHeadP=0x%08x; hwNextED=0x%08x;\n", s,  ed->ed_dma, le32_to_cpu(ed->hwINFO), le32_to_cpu(ed->hwTailP),le32_to_cpu(ed->hwHeadP),le32_to_cpu(ed->hwNextED));
 return;
}
void ohci_dump_data(unsigned char *addr, unsigned int len)
{int i,j;
 diag_printf("%s> len= 0x%02x\n", __FUNCTION__, len);
 for(i=0; i < len; i+=32)
   	{diag_printf("\n %02x> ", i);
   	 for(j=0; j<32; j++)
       	diag_printf("%02x ", addr[i+j]);
   	}
 diag_printf("\n ");
 return;
}
//#endif
/*=====================================================================================================*/
static void*  pciwindow_mem_alloc(unsigned int  size, unsigned int mask)
{void *p_memory;
 unsigned int v, _size = size;
 if(!( (USB_PCI_MEM_MAP_BASE <= (unsigned int)usb_heap_free) && ((USB_PCI_MEM_MAP_BASE +  USB_PCI_MEM_MAP_SIZE) > (unsigned int)usb_heap_free)
        && (0 < usb_heap_size) && (USB_PCI_MEM_MAP_SIZE >= usb_heap_size) && (USB_PCI_MEM_MAP_BASE == (unsigned int)usb_heap_base)) )
			{diag_printf("\t - mem alloc error.\n"); return (void *)0;}
 p_memory = (void *)0;
 if ( (usb_heap_free+size) < (usb_heap_base+usb_heap_size) )
    {unsigned int *p;
     v=((unsigned int)usb_heap_free)+(mask -1);
     v &= ~(mask -1);
     p_memory = (void *)v;
     v= (v + size + mask -1) & ~(mask -1);
     usb_heap_free = (void *)v;
     for ( p = (unsigned int *)p_memory; _size > 0; _size -= 4 ) *p++ = 0;
    }
 if(NULL == p_memory)
	{diag_printf("\t - memory allocation error.\n" );return NULL;}
 memset(p_memory, 0, size);
 return p_memory;
}
/*=====================================================================================================*/
static int ohci_mem_init (struct ohci_hc *ohci)
{int i;
 usb_buf_0=(unsigned int *)(USB_PCI_MEM_MAP_BASE);
 usb_buf_2=(unsigned int *)(USB_PCI_MEM_MAP_BASE + 2048U);
 usb_buf_1=(unsigned int *)(USB_PCI_MEM_MAP_BASE + 4096U);
 if((ohci->hcca = (struct ohci_hcca *)pciwindow_mem_alloc(sizeof(struct ohci_hcca), 256))==NULL){return 0;}
 memset(ohci->hcca, 0, sizeof(struct ohci_hcca));
 ed_0= (struct ed *)pciwindow_mem_alloc(sizeof(struct ed), 64);
 for(i=0; i< USB_MAX_OHCI_TD; i++)
	{if((ohci->td_pool[i]= (struct td *)pciwindow_mem_alloc(sizeof(struct td), 64))==NULL)return 0;}
 return 1;
}
/*=====================================================================================================*/
static int ohci_reset (struct ohci_hc * ohci)
{int timeout = 30;
 int smm_timeout = 500; /* 0,5 sec */
 /* PA-RISC doesn't have SMM, but PDC might leave IR set */
 if (ohci_readl(&ohci->regs->control) & OHCI_CTRL_IR)
	{ /* SMM owns the HC */
	 ohci_writel(OHCI_INTR_OC, &ohci->regs->intrenable);
	 ohci_writel (OHCI_OCR, &ohci->regs->cmdstatus); /* request ownership */
	 diag_printf("USB HC TakeOver from SMM");
	 while (ohci_readl (&ohci->regs->control) & OHCI_CTRL_IR)
		{CYGACC_CALL_IF_DELAY_US (10000);
		 if (--smm_timeout == 0)
			{diag_printf("USB HC TakeOver failed!"); return -1;}
		}
	 writel(OHCI_CTRL_RWC, &ohci->regs->control);
	}
 /* Disable HC interrupts */
 ohci_writel(0xffffffffU, &ohci->regs->intrdisable);
 ohci_writel(0xffffffffU, &ohci->regs->intrstatus);
 /* Reset USB (needed by some controllers) */
 ohci_writel (0, &ohci->regs->control);
 /* Force a state change from USBRESET to USBOPERATIONAL for ALi */
 (void) ohci_readl (&ohci->regs->control);	/* PCI posting */
 /* HC Reset requires max 10 ms delay */
 ohci_writel (OHCI_HCR,  &ohci->regs->cmdstatus);
 while ((ohci_readl (&ohci->regs->cmdstatus) & OHCI_HCR) != 0) 
	{if (--timeout == 0) 
		{diag_printf(" - ohci reset timed out.\n");return -1;}
	 CYGACC_CALL_IF_DELAY_US (1);
	}
#ifdef OHCI_DEBUG
 diag_printf(" - ohci reset ok.\n");
#endif
 return 0;
}
#define	PORT_RESET_MSEC		20
#define	PORT_RESET_HW_MSEC	20
/*=====================================================================================================*/
static void root_hub_port_reset (struct ohci_hc *ohci, unsigned port)
{unsigned int volatile *port_status;
 unsigned int v;

 port_status = &ohci->regs->roothub.portstatus [port];
 while (1) 
	{v = ohci_readl (port_status);
#ifdef OHCI_DEBUG
	 diag_printf("%s> [%d] status= 0x%08x.\n", __FUNCTION__, __LINE__, v);
#endif
	 if (!(v & RH_PS_PRS))
		break;
	 CYGACC_CALL_IF_DELAY_US (500);
	}
 if (!(v & RH_PS_CCS))
	{return;}
 if (v & RH_PS_PRSC)
	ohci_writel (RH_PS_PRSC, port_status);
 ohci_writel (RH_PS_PRS, port_status);	 /* start the next reset, sleep till it's probably done */
 mdelay(PORT_RESET_HW_MSEC);
 while (1) 
	{CYGACC_CALL_IF_DELAY_US (500);
     v = ohci_readl (port_status);
#ifdef OHCI_DEBUG
	 diag_printf("%s> [%d] status= 0x%08x.\n", __FUNCTION__, __LINE__, v);
#endif
	 if (v & RH_PS_PRSC)
		{ohci_writel (RH_PS_PRSC, port_status); return;}
	}
 return;
}
/*=====================================================================================================*/
/* Start an OHCI controller, set the BUS operational, disable interrupts, connect the virtual root hub */
static int ohci_start (struct ohci_hc * ohci)
{unsigned int v;
 unsigned int fminterval;
 /* Tell the controller where the control and bulk lists are The lists are empty now. */
 ohci_writel (0, &ohci->regs->ed_controlhead);
 ohci_writel (0, &ohci->regs->ed_bulkhead);
 v= CYGARC_PCI_DMA_ADDRESS(ohci->hcca);
 ohci_writel (v, &ohci->regs->hcca); /* a reset clears this */
 fminterval = 0x2edf;
 ohci_writel ((fminterval * 9) / 10, &ohci->regs->periodicstart);
 fminterval |= ((((fminterval - 210) * 6) / 7) << 16); 
 ohci_writel (fminterval, &ohci->regs->fminterval);	
 ohci_writel (0x628, &ohci->regs->lsthresh);
 /* start controller operations */
 v = (OHCI_CTRL_CBSR & 0x3) | OHCI_USB_OPER;
 ohci_writel (v, &ohci->regs->control);
 /* Choose the interrupts we care about now, others later on demand */
 ohci_writel (0xffffffff, &ohci->regs->intrdisable);
 ohci_writel ((OHCI_INTR_WDH | OHCI_INTR_UE), &ohci->regs->intrenable);
 ohci_writel (0xffffffff, &ohci->regs->intrstatus);
 v= roothub_a (ohci);
 ohci_writel ((v | RH_A_NPS) & ~RH_A_PSM, &ohci->regs->roothub.a);
 ohci_writel (0, &ohci->regs->roothub.b);
 ohci_writel (RH_HS_LPSC, &ohci->regs->roothub.status);
 (void)ohci_readl (&ohci->regs->intrenable); /* PCI posting flush */
 // POTPGT delay is bits 24-31, in 2 ms units.
 CYGACC_CALL_IF_DELAY_US (50000);
 /* connect the virtual root hub */
// ohci->rh.devnum = 0;
 for (v = 0; v < ohci->num_ports; v++)
	{ohci_writel (RH_PS_PPS, &ohci->regs->roothub.portstatus [v]);}
 ohci->enabled=1;
 return 0;
}
/*=====================================================================================================*/
static int ohci_init_find_root_hub (struct ohci_hc *ohci)
{unsigned int control, status, i, j, n;
 const char *_WNODEV={" - device not present.\n"};
 int r;
 control= ohci_readl(&ohci->regs->control);
 ohci->num_ports = roothub_a(ohci) & RH_A_NDP;
 if ( (((control & OHCI_CTRL_HCFS) != OHCI_USB_OPER) || !(ohci->enabled))) 
	{return 0;}
// if (roothub_status (ohci) & (RH_HS_LPSC | RH_HS_OCIC))
//	{;}
 usb_dev_found=0;
 {struct ohci_hc *o;
  for (n= i = 0; i < ohci->index; i++)
	{o= &ohci_array[i];
	 n+= o->num_ports;
	}
 }
 for (i = 0; i < ohci->num_ports; i++) 											/* look at each port 		*/
	{diag_printf("   USB bus %d scan . . .", n + i);
	 status = roothub_portstatus (ohci, i);
	 if(status & (RH_PS_CSC))													/* connect status change 	*/
		{ohci_writel(RH_PS_CSC, &ohci->regs->roothub.portstatus [i]);
		 if(!(status & RH_PS_CCS))												/* device disconnected   	*/
			{diag_printf("%s",_WNODEV);
	 	 	 continue;
			}
 		 for(j=0; j< 4; j++)													/* device connected   		*/
			{mdelay (100);
	 		 status= ohci_readl(&ohci->regs->roothub.portstatus [i]);
	 		 if(status & RH_PS_CSC)
				{ohci_writel(RH_PS_CSC, &ohci->regs->roothub.portstatus [i]); j=0; continue;}
			}
		 status = roothub_portstatus (ohci, i);
		 root_hub_port_reset (ohci, i);
		 status = roothub_portstatus (ohci, i);
		 usb_dev[usb_dev_found].hc= ohci;
		 usb_dev[usb_dev_found].devnum= 0;
		 if((r= usb_new_device( &usb_dev[usb_dev_found], usb_dev_found+ 2)) != ENOERR)
			{return -1;}
		 usb_dev_found++;
		 continue;
		}
	 diag_printf("%s", _WNODEV);
	}
 return i;
}
/*=====================================================================================================*/
static int ohci_check_root_hub (struct ohci_hc *ohci)
{unsigned int control, status, i;
 control= ohci_readl(&ohci->regs->control);
 if ( (((control & OHCI_CTRL_HCFS) != OHCI_USB_OPER) || !(ohci->enabled))) 
	{return 0;}
 for (i = 0; i < ohci->num_ports; i++) 											/* look at each port 		*/
	{status = roothub_portstatus (ohci, i);
	 if(status & (RH_PS_CSC))													/* connect status change 	*/
		{if(!(status & RH_PS_CCS))												/* device disconnected   	*/
		 	{diag_printf("\t - error: usb device disconnected.\n");return 0;}
		 diag_printf("\t - new device connected. System restart is needed.\n");
		 return 0;
		}
	}
 return 1;
}
/*=====================================================================================================*/
/* enqueue next TD for this URB (OHCI spec 5.2.8.2) */
static void td_fill (struct ohci_hc *ohci, unsigned int info, unsigned int data, int len, struct ed *ed, int index, int td_flag)
{struct td	*td, *empty_td;
 unsigned int  v;
 // ASSERT (index < urb_priv->length);
 /* aim for only one interrupt per urb.  mostly applies to control and iso; other urbs rarely need more than one TD per urb.
  * this way, only final tds (or ones with an error) cause IRQs. at least immediately; use DI=6 in case any control request is
  * tempted to die part way through.  (and to force the hc to flush its donelist soonish, even on unlink paths.)
  *
  * NOTE: could delay interrupts even for the last TD, and get fewer interrupts ... increasing per-urb latency by sharing interrupts.
  * Drivers that queue bulk urbs may request that behavior. */
 td= ohci->td_pool[index];
 memset(td, 0, sizeof(struct td));
 {empty_td= ohci->td_pool[index+1];
  empty_td->hwINFO= empty_td->hwCBP= empty_td->hwBE= 0x00000000;
  empty_td->td_dma= CYGARC_PCI_DMA_ADDRESS(empty_td);
  empty_td->hwNextTD=cpu_to_le32(empty_td->td_dma);
  empty_td->ed= ed;
  empty_td->data_dma= 0x00000000;
}
 ohci->td_pending= index+1;
 ed->td_pending= index+1;
 td->index= index;
 td->ed= ed;
 td->td_dma= CYGARC_PCI_DMA_ADDRESS(td);
 td->data_dma= data;
 if(!(td_flag & OHCI_LAST_TD))
 	{info |= TD_DI_SET(6);}
 td->hwINFO= cpu_to_le32 (info);
 if (!len)
	{data = 0;}
 td->hwCBP = cpu_to_le32(data);
 td->hwNextTD= cpu_to_le32(empty_td->td_dma);
 td->hwBE = 0;
 if (data)
	{v= data + len - 1;
	 td->hwBE = cpu_to_le32 (v);}
 v= cpu_to_le32(td->td_dma);
 if(index)															/* link into td queue */
	{ohci->td_pool[index-1]->hwNextTD= v;}
 else
	{ed->hwHeadP= v;}
 ed->hwTailP = td->hwNextTD;
#ifdef OHCI_DEBUG
 diag_printf ("%s> td%d_dma=0x%08x\n", __FUNCTION__, index, td->td_dma);
#endif
 ohci_sync (ohci);
 return;
}
/*=====================================================================================================*/
static void td_done (struct ohci_hc *ohci, struct urb *urb, struct td *td)
{unsigned int tdINFO, tdBE;
 int cc = 0, type;
 type  = usb_pipetype (urb->pipe);
 tdINFO= le32_to_cpup (&td->hwINFO);
 tdBE  = le32_to_cpup (&td->hwBE);
 cc   = TD_CC_GET (tdINFO);
 /* update packet status if needed (short is normally ok) */
 if (cc == TD_DATAUNDERRUN	&& !(urb->transfer_flags & URB_SHORT_NOT_OK))
	{cc = TD_CC_NOERROR;}
 if (cc != TD_CC_NOERROR && cc < 0x0E) 
	{urb->status = cc_to_error [cc];}
 /* count all non-empty packets except control SETUP packet */
 if ((type != PIPE_CONTROL || td->index != 0) && tdBE != 0) 
	{if (td->hwCBP == 0)
		urb->actual_length += tdBE - td->data_dma + 1;
	 else
		urb->actual_length += le32_to_cpup (&td->hwCBP)- td->data_dma;
	}
 if (cc != TD_CC_NOERROR && cc < 0x0E)
	{diag_printf("urb td(%d) cc= %d, len=%d/%d\n", 1 + td->index, cc, urb->actual_length, urb->transfer_buffer_length);}
#ifdef OHCI_DEBUG
 diag_printf("%s> [%d] actual_length=0x%04x\n", __FUNCTION__, __LINE__, urb->actual_length);
#endif
 return;
}
/*=====================================================================================================*/
static int process_done_list (struct ohci_hc *ohci)
{struct td *td;
 struct ed *ed;
 struct urb *urb;
 unsigned int i, td_dma, cc, td_done_index;
 urb= ohci->urb;
 ed= ohci->ed;
 td= ohci->td_pool[ohci->td_pending -1];
 td_dma = le32_to_cpup (&ohci->hcca->done_head);
 if(td_dma != td->td_dma)
 	{for(i=1000; 1; i--)
		{td_dma = le32_to_cpup (&ohci->hcca->done_head);
		 if(td_dma == td->td_dma)
			{break;}
	 	 CYGACC_CALL_IF_DELAY_US(10);
	 	 if(i)
			{continue;}
 	 	 diag_printf("%s> [%d] error: donehead=0x%08x!=0x%08x\n", __FUNCTION__, __LINE__, td_dma,td->td_dma);
	 	 return 0;
	}	}
 td_dma = le32_to_cpup (&ohci->hcca->done_head);
 ohci->hcca->done_head = 0;
#ifdef OHCI_DEBUG
 diag_printf("%s> [%d] done_head=0x%08x \n", __FUNCTION__, __LINE__, td_dma);
#endif
 if(!td_dma)
	{diag_printf("%s> [%d] error \n", __FUNCTION__, __LINE__); return 0;}
 ohci_sync(ohci);
 /* get TD from hc's singly linked list, and prepend to ours.  ed->td_list changes later. */
 for(td_done_index=0; ; td_done_index++)
	 {if(td_done_index > ohci->td_pending)
		{diag_printf("%s> [%d] error[td done= %d/%d].\n", __FUNCTION__, __LINE__, td_done_index, ohci->td_pending);return 0;}
	  td= ohci->td_pool[td_done_index];
	  td->hwINFO |= cpu_to_le32 (TD_DONE);
	  if(td_dma == td->td_dma)		
	    {if(td_done_index != (ohci->td_pending - 1))
		 	{diag_printf("%s> [%d] exchane error [%d/%d transactions done].\n", __FUNCTION__, __LINE__, td_done_index, ohci->td_pending -1);return 0;}
		 cc = TD_CC_GET (le32_to_cpup (&td->hwINFO));
		 /* Non-iso endpoints can halt on error; un-halt, and dequeue any other TDs from this urb. No other TD could have caused the halt. */
		 if((cc != TD_CC_NOERROR) && (ed->hwHeadP & cpu_to_le32(ED_H)))
			{diag_printf("%s> [%d] td error 0x%x.\n", __FUNCTION__, __LINE__, cc);}
		 break;
	}	}	
 for(i=0; i <= td_done_index; i++) 
	{td= ohci->td_pool[i];
	 td_done (ohci, urb, td);
  	}
 if (urb->status == -EINPROGRESS)
	{urb->status = 0;}
 return 1;
}
/*=====================================================================================================*/
/* link an ed into one of the HC chains */
static int ed_schedule (struct ohci_hc *ohci, struct ed *ed)
{unsigned int control, ep_type;

 ohci->ed= ed;
 ohci_sync (ohci);
 /* we care about rm_list when setting CLE/BLE in case the HC was at work on some TD when CLE/BLE was turned off, and isn't quiesced yet.  finish_unlinks() restarts as needed, some upcoming INTR_SF.
  * control and bulk EDs are doubly linked (ed_next, ed_prev), but periodic ones are singly linked (ed_next). that's because the periodic schedule encodes a tree like figure 3-5 in the ohci
  * spec:  each qh can have several "previous" nodes, and the tree doesn't have unused/idle descriptors. */
 control= ohci_readl(&ohci->regs->control);
 ep_type= usb_pipetype(ed->pipe);
// ed->hwINFO &= ~cpu_to_le32 (ED_SKIP);
 ohci->hcca->done_head = 0;
 ohci_writel (0xffffffffU, &ohci->regs->intrstatus);
 switch (ep_type) 
	{case PIPE_CONTROL:		if(control & OHCI_CTRL_CLE) {diag_printf("%s> [%d] warning: control list enable.\n", __FUNCTION__, __LINE__);}
							ohci_writel (ed->ed_dma, &ohci->regs->ed_controlhead);
							if (!(control & OHCI_CTRL_CLE))
								{ohci_sync(ohci);
								 control |= OHCI_CTRL_CLE;
								 ohci_writel (0, &ohci->regs->ed_controlcurrent);
								 ohci_writel (control, &ohci->regs->control);
								}
							return 1;
	  case PIPE_BULK:		if(control & OHCI_CTRL_BLE) {diag_printf("%s> [%d] warning: bulk list enable.\n", __FUNCTION__, __LINE__);}
						 	ohci_writel (ed->ed_dma, &ohci->regs->ed_bulkhead);
							if (!(control & OHCI_CTRL_BLE)) 
								{ohci_sync(ohci);
								 control |= OHCI_CTRL_BLE;
								 ohci_writel (0, &ohci->regs->ed_bulkcurrent);
								 ohci_writel (control, &ohci->regs->control);
								}
							return 1;
	  default:				diag_printf("%s> [%d] wrong endpoint type.\n", __FUNCTION__, __LINE__);
	}
 /* the HC may not see the schedule updates yet, but if it does then they'll be properly ordered.*/
 return 0;
}
/*=====================================================================================================*/
static int ed_deschedule (struct ohci_hc *ohci, struct ed *ed) 
{unsigned int control;

 ed->hwINFO |= cpu_to_le32 (ED_SKIP);
 ohci->ed= NULL;
 control= ohci_readl(&ohci->regs->control) & ~(OHCI_CTRL_CLE | OHCI_CTRL_BLE | OHCI_CTRL_PLE | OHCI_CTRL_IE);
 ohci_writel(control, &ohci->regs->control);
 ohci_writel (0, &ohci->regs->ed_controlcurrent);
 ohci_writel (0, &ohci->regs->ed_bulkcurrent);
 ohci_writel (0, &ohci->regs->ed_controlhead);
 ohci_writel (0, &ohci->regs->ed_bulkhead);
 return 0;
}
/*=====================================================================================================*/
static struct ed *init_ed(struct ohci_hc *ohci, struct urb *urb, struct ed *ed)
{struct usb_device *dev;
 unsigned int info;
 unsigned int pipe, endpoint, is_out;
 pipe= urb->pipe;
 dev= urb->dev;

 endpoint= usb_pipeendpoint(pipe) & ~USB_DIR_IN;
 is_out= usb_pipeout(pipe);
 memset((char *) ed, 0, sizeof(struct ed));
 td_fill (ohci, TD_CC | TD_DP_SETUP | TD_T_DATA0, 0, 0, ed, 0,1);		/* make empty td 							*/
// ed->hwINFO  = cpu_to_le32 (ED_SKIP);
 info = usb_pipedevice (pipe);
 info|= endpoint << 7;
// if (udev->speed == USB_SPEED_LOW)
//	{info |= ED_LOWSPEED;}
 /* only control transfers store pids in tds */
 if (usb_pipetype(pipe) == PIPE_BULK) 
	{info |= (is_out)? ED_OUT : ED_IN;}
 info|= usb_maxpacket (dev, endpoint, is_out) << 16;
 ed->hwINFO  = cpu_to_le32(info);
 ed->hwTailP = cpu_to_le32(ohci->td_pool[0]->td_dma);				/* link epty td 							*/
 ed->hwHeadP = ed->hwTailP;
 ed->hwNextED= 0;
 ed->ed_dma  = CYGARC_PCI_DMA_ADDRESS(ed);
 ed->pipe= pipe;													/* add & link new endpoint 					*/
 ed->td_pending=0;
 ohci->ed= ed;
 ed->dev= dev;
 return ed;
}
/*=====================================================================================================*/
static int ohci_check_irq (struct ohci_hc *ohci)
{struct ohci_regs *regs = ohci->regs;
 unsigned int iir, ier, i; 
 iir = ohci_readl (&regs->intrstatus);
 ier= ohci_readl (&regs->intrenable);
 if (iir == 0xffffffffU) 
	{ohci->enabled=0;
	 diag_printf("- device removed.\n");							/* interrupt for some other device? */
	 return -1;
	}
 if (iir & OHCI_INTR_UE) 											/* unrecoverable error 	 e.g. due to PCI Master/Target Abort */
	{ohci->enabled=0;
	 diag_printf("- OHCI Unrecoverable Error, disabled\n");
	 ohci_reset (ohci);
	 return -1;
	}
 iir &= ier;
 if (iir == 0) 
	{return 0;} 
 if (iir & OHCI_INTR_WDH) 											/* writeback of done_head */
	{ohci_writel (0xffffffff, &ohci->regs->intrstatus);
	 i= process_done_list (ohci);
	 ohci_writel (0xffffffff, &regs->intrstatus);
	 ohci_writel ((OHCI_INTR_WDH | OHCI_INTR_UE), &regs->intrenable); 
	 (void) ohci_readl (&ohci->regs->control);						/* flush those writes */
	 if(!i)
		{return -1;}
	 return 1;
	}
 ohci_writel (0xffffffff, &regs->intrstatus);
 (void) ohci_readl (&ohci->regs->control);						/* flush those writes */
 diag_printf ("- no enabled irq\n");
 return 0;
}
/*=====================================================================================================*/
/* Prepare all TDs of a transfer, and queue them onto the ED.
 * Caller guarantees HC is active.
 * Usually the ED is already on the schedule, so TDs might be processed as soon as they're queued. */
int ohci_do_urb (struct urb *urb) 
{struct ohci_hc  *ohci;
// struct ed       ed;
 unsigned int    pipe, data, info = 0;
 int             i, r, data_len, is_out, td_index= 0/*, periodic= 0*/;

 ohci= urb->dev->hc;
 if(!ohci_check_root_hub (ohci))
	{return -ECOMM;}
 pipe= urb->pipe;
 if (!ohci->enabled) 
	{return -ESHUTDOWN;}
 init_ed(ohci, urb, ed_0);
 data_len= urb->transfer_buffer_length;
 is_out= usb_pipeout(pipe);
 data = 0;
 if (data_len)
	{data = urb->transfer_dma;}
 /* NOTE:  TD_CC is set so we can tell which TDs the HC processed by using TD_CC_GET, as well as by seeing them on the done list.
  * (CC = NotAccessed ... 0x0F, or 0x0E in PSWs for ISO.) */
 switch (usb_pipetype(pipe)) 
	{/* Bulk and interrupt are identical except for where in the schedule their EDs live. */
	 case PIPE_BULK:		info= (is_out)? (TD_T_TOGGLE | TD_CC | TD_R | TD_DP_OUT) : (TD_T_TOGGLE | TD_CC | TD_R | TD_DP_IN);
							/* TDs _could_ transfer up to 8K each */
							for ( ;data_len > 4096; td_index++)
								{td_fill (ohci, info, data, 4096, ed_0, td_index, 0);
								 data += 4096;
								 data_len -= 4096;
								}
							/* maybe avoid ED halt on final TD short read */
							td_fill (ohci, info, data, data_len, ed_0, td_index, OHCI_LAST_TD);
							td_index++;
//							if ((urb->transfer_flags & URB_ZERO_PACKET)	&& (td_index < urb->length)) 
//								{td_fill (ohci, info, 0, 0, ed_0, td_index);td_index++;}
							/* maybe kickstart bulk list */
 							if(usb_gettoggle(ed_0->dev, is_out))
								{ed_0->hwHeadP|= cpu_to_le32(ED_C);}
#ifdef OHCI_DEBUG
							ohci_dump__ed(ed_0, "ed_0");
							if(is_out)
								ohci_dump_data((unsigned char *)urb->transfer_buffer, data_len);
#endif
 							if((r=ed_schedule(ohci, ed_0)) < 0)
								return r;
							ohci_writel (OHCI_BLF, &ohci->regs->cmdstatus);						/* bulk list filled */
							break;
	/* control manages DATA0/DATA1 toggle per-request; SETUP resets it, any DATA phase works normally, and the STATUS ack is special. */
	 case PIPE_CONTROL:		info = TD_CC | TD_DP_SETUP | TD_T_DATA0;
							td_fill (ohci, info, urb->setup_dma, 8, ed_0, td_index, 0);
							td_index++;
							if (data_len > 0)
								{info = TD_CC | TD_R | TD_T_DATA1;
								 info |= (is_out)? (TD_DP_OUT) : (TD_DP_IN);
								 td_fill (ohci, info, data, data_len, ed_0, td_index, 0);			/* NOTE:  mishandles transfers >8K, some >4K */
								 td_index++;
								}
							info = ((is_out) || (data_len == 0))? (TD_CC | TD_DP_IN | TD_T_DATA1) : (TD_CC | TD_DP_OUT | TD_T_DATA1);
							td_fill (ohci, info, data, 0, ed_0, td_index, 1);
							td_index++;
#ifdef OHCI_DEBUG
	 						ohci_dump__ed(ed_0, "ed_0");
#endif
							/* maybe kickstart control list */
 							if((r=ed_schedule(ohci, ed_0)) < 0)
								return r;
							ohci_writel (OHCI_CLF, &ohci->regs->cmdstatus);						/* control list filled */
							break;
	 default:				diag_printf("%s> [%d] wrong urb type.\n", __FUNCTION__, __LINE__);
							return -1;
	}
 ohci->urb= urb;
 for(i=0; i < 0x2000; i++)
	{
#ifdef OHCI_DEBUG
	 diag_printf("%s> [%d].\n", __FUNCTION__, __LINE__);
	 ohci_dump_ed(ed_0, "ed");
	 for(r=0; r < td_index; r++)
		ohci_dump_td(ohci->td_pool[r], "td");
#endif
	 if((r=ohci_check_irq (ohci)) == 0)
		{CYGACC_CALL_IF_DELAY_US (1000);continue;}
	 ed_deschedule (ohci, ed_0);
 	 ohci->urb= NULL;
	 if(r < 0)
		{diag_printf("%s> [%d] urb run error.\n", __FUNCTION__, __LINE__);return r;}
#ifdef OHCI_DEBUG
	 diag_printf("%s> [%d] urb done.\n", __FUNCTION__, __LINE__);
#endif
	 return 0;
	}
 ed_deschedule (ohci, ed_0);
 ohci->urb= NULL;
 diag_printf("%s> [%d] urb run error.\n", __FUNCTION__, __LINE__);
 return -1;
}
#define PCI_CLASS_SERIAL_USB_OHCI	0x0c0310
/*=====================================================================================================*/
static cyg_bool find_ohci_match_func( unsigned short v, unsigned short d, unsigned int c, void *p )
{int usb_ohci;
/* int isp1561_chip;
 isp1561_chip= (0x1131 == v) && ((0x1561 == d)) */
 usb_ohci= (c == PCI_CLASS_SERIAL_USB_OHCI);
 return  usb_ohci;
}
/*=====================================================================================================*/
int usb_init_find_ohci( void )
{cyg_pci_device dev_info;
 unsigned int base_addr, devid;
 unsigned short cmd;
 struct ohci_hc *ohci;
 usb_heap_size = USB_PCI_MEM_MAP_SIZE;
 usb_heap_base = (unsigned char *)(USB_PCI_MEM_MAP_BASE);
 usb_heap_free = (unsigned char *)(USB_PCI_MEM_MAP_BASE + USB_BUFS_SIZE);

 memset(ohci_array, 0, sizeof(struct ohci_hc) * USB_MAX_OHCI);
 memset(&usb_dev, 0, sizeof(struct usb_device) *USB_MAXDEV);
 devid = CYG_PCI_NULL_DEVFN;
 for (ohci_found = 0; ohci_found < USB_MAX_OHCI; ohci_found++) 
	{diag_printf("USB%d controller. . .\n", ohci_found);
	 if(!cyg_pci_find_matching( &find_ohci_match_func, NULL, &devid ))
		{diag_printf("\t - .\n"); return ohci_found;}
	 ohci= &ohci_array[ohci_found];
	 memset(ohci, 0, sizeof(struct ohci_hc));
     ohci->index= ohci_found;
	 cyg_pci_get_device_info(devid, &dev_info);
     if(!cyg_pci_configure_device(&dev_info))
		{diag_printf("\t - configure error.\n");continue;}
#ifdef OHCI_DEBUG
     diag_printf("Found device on bus %d, devfn 0x%02x:\n", CYG_PCI_DEV_GET_BUS(devid), CYG_PCI_DEV_GET_DEVFN(devid));
     if (dev_info.command & CYG_PCI_CFG_COMMAND_ACTIVE)
          { }
     diag_printf(" Vendor    0x%04x", dev_info.vendor);
     diag_printf("\n\tDevice    0x%04x", dev_info.device);
     diag_printf("\n\tCommand   0x%04x, Status 0x%04x\n", dev_info.command, dev_info.status);
     diag_printf(" Class/Rev 0x%08x", dev_info.class_rev);
     diag_printf("\n\tHeader 0x%02x\n", dev_info.header_type);
     diag_printf("\tSubVendor 0x%04x, Sub ID 0x%04x\n",dev_info.header.normal.sub_vendor, dev_info.header.normal.sub_id);
     diag_printf("\tbar[0]    0x%08x /", dev_info.base_address[0]);
     diag_printf(" probed size 0x%08x / CPU addr 0x%08x\n", dev_info.base_size[0], dev_info.base_map[0]);
#endif
	 base_addr= dev_info.base_map[0];
	 ohci->regs= (struct ohci_regs *)base_addr;
     cyg_pci_read_config_uint16(dev_info.devid,CYG_PCI_CFG_COMMAND, &cmd);
     cmd |= (CYG_PCI_CFG_COMMAND_IO | CYG_PCI_CFG_COMMAND_MEMORY); // enable bus master
     cyg_pci_write_config_uint16(dev_info.devid, CYG_PCI_CFG_COMMAND, cmd);
//	 ohci_start (ohci);
	 if (ohci_reset (ohci) < 0)
		{continue;}
	 ohci->num_ports = roothub_a(ohci) & RH_A_NDP;
     cmd |= (CYG_PCI_CFG_COMMAND_MASTER); // enable bus master
     cyg_pci_write_config_uint16(dev_info.devid, CYG_PCI_CFG_COMMAND, cmd);
     ohci->devid = devid;
	 if(!ohci_mem_init (ohci))
		{continue;}
	 ohci_start (ohci);
	 ohci_init_find_root_hub (ohci);
//	 root_hub_port_reset (ohci, 0);
	}
 return ohci_found;
}







