#include <sys/types.h>
#include <unistd.h>
#include <error.h>
#include <errno.h>

#include <dlfcn.h>

#ifndef LIBCPATH
#define LIBCPATH "/lib/libc.so.6"
#endif

static void *
get_func (const char *symbol)
{
	static void *handle = 0;
	void   *addr;

	if (!handle)
		handle = dlopen (LIBCPATH, RTLD_LAZY);
	if (!handle)
		error (1, errno, "dlopen");
	addr = dlsym (handle, symbol);
	if (!addr)
		error (1, errno, "dlsym: %s", symbol);
	return addr;
}

typedef int (*chown_fptr) (const char *, uid_t, gid_t);

int
chown (const char *path, uid_t owner, gid_t group)
{
	if (getuid ())
	{
		static chown_fptr func = 0;

		if (!func)
			func = get_func ("chown");
		return func (path, owner, group);
	}
	return 0;
}

typedef int (*fchown_fptr) (int, uid_t, gid_t);

int
fchown (int fd, uid_t owner, gid_t group)
{
	if (getuid ())
	{
		static fchown_fptr func = 0;

		if (!func)
			func = get_func ("fchown");
		return func (fd, owner, group);
	}
	return 0;
}

typedef int (*lchown_fptr) (const char *, uid_t, gid_t);

int
lchown (const char *path, uid_t owner, gid_t group)
{
	if (getuid ())
	{
		static lchown_fptr func = 0;

		if (!func)
			func = get_func ("lchown");
		return func (path, owner, group);
	}
	return 0;
}

typedef int (*chmod_fptr) (const char *, mode_t);

int
chmod (const char *path, mode_t mode)
{
	if (getuid ())
	{
		static chmod_fptr func = 0;

		if (!func)
			func = get_func ("chmod");
		return func (path, mode);
	}
	return 0;
}

typedef int (*fchmod_fptr) (int, mode_t);

int
fchmod (int fildes, mode_t mode)
{
	if (getuid ())
	{
		static fchmod_fptr func = 0;

		if (!func)
			func = get_func ("fchmod");
		return func (fildes, mode);
	}
	return 0;
}
