/*
	Copyright (c) 2000-2008 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the tagran keyboard event handler.

   Tagran is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.

   Tagran is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include<assert.h>

#include<iostream>
#include<fstream>
#include<list>
#include<vector>

#include<errno.h>
#include<sys/types.h>
#include<unistd.h>
#include<fcntl.h>
#include<sys/stat.h>
#include<sys/wait.h>
#include<time.h>
#include<sys/time.h>

#define VERSION "1.0"
#define COPYRIGHT "Copyright (c) 2000-2008 Michael Pozhidaev <msp@altlinux.org>"
#define DEFAULT_CONFIG "/etc/tagran.conf"
#define ERROR_PREFIX "error:"
#define SHELL "/bin/sh"

#define LOG_ERROR 100

struct InputEvent 
{
  struct timeval time;
  unsigned short type;
  unsigned short code;
  unsigned int value;
}; //struct InputEvent;

std::string trim(const std::string& s)
{
  int l = -1, r = -1, i;
  for(i = 0;i < s.length();i++)
    {
      if (l == -1 && s[i] != ' ' && s[i] != '\t')
	l = i;
      if (s[i] != ' ' && s[i] != '\t')
	r = i;
    }
  if (r < 0)
    return std::string();
  assert(l >= 0);
  assert(l <= r);
  std::string n;
  for(i = l;i <= r;i++)
    n += s[i];
  return n;
}

bool checkInt(const std::string& s)
{
  for(int i = 0;i < s.length();i++)
    if (s[i] < '0' || s[i] > '9')
      return 0;
  return 1;
}

int parseInt(const std::string& s)
{
  int k = 0;
  for(int i = 0;i < s.length();i++)
    {
      assert(s[i] >= '0' && s[i] <= '9');
      k *= 10;
      k += (s[i] - '0');
    }
  return k;
}

std::string getErrorMessage(int code)
{
  return sys_errlist[code];
}

void logMessage(int level, const std::string& message)
{
  std::cerr << message << std::endl;//FIXME:
}

class SystemException
{
public:
  SystemException() : m_code(errno) {}
  SystemException(const std::string& comment): m_code(errno), m_comment(comment) {}
  SystemException(int code, const std::string& comment): m_code(code), m_comment(comment) {}

  int getCode() const
  {
    return m_code;
  }

  std::string getComment() const
  {
    return m_comment;
  }

private:
  int m_code;
  std::string m_comment;
}; //class SystemException;

class InputEventReader
{
public:
  InputEventReader(const std::string& deviceName): m_deviceName(deviceName), m_handler(-1) {}
  virtual ~InputEventReader() {close();}

  std::string getDevicename() const
  {
    return m_deviceName;
  }

  void open()
  {
    assert(m_handler == -1);
    assert(!m_deviceName.empty());
    m_handler = ::open(m_deviceName.c_str(), O_RDONLY);
    if (m_handler == -1)
      throw SystemException("open(" + m_deviceName + ")");
  }

  void close()
  {
    if (m_handler == -1)
      return;
    ::close(m_handler);
    m_handler = -1;
  }

  InputEvent readNextEvent()
  {
    assert(m_handler != -1);
    InputEvent e;
    int r = ::read(m_handler, &e, sizeof(InputEvent));
    if (r != sizeof(InputEvent))
      throw SystemException("read()");
    return e;
  }

private:
  std::string m_deviceName;
  int m_handler;
}; //class InputEventReader;

class EventCommand
{
public:
  EventCommand(): m_type(-1), m_code(-1), m_value(-1) {}
  EventCommand(const std::string& command, int type = -1, int code = -1, int value = -1): m_command(command), m_type(type), m_code(code), m_value(value) {}

  std::string getCommand() const
  {
    return m_command;
  }

  int getType() const
  {
    return m_type;
  }

  int getCode() const
  {
    return m_code;
  }

  int getValue() const
  {
    return m_value;
  }

private:
  std::string m_command;
  int m_type, m_code, m_value;
}; //class eventCommand;

typedef std::list<EventCommand> EventCommandList;

class ConfigFileIterator
{
public:
  ConfigFileIterator(std::istream& stream): m_stream(stream), m_line(0) {}

  int getLine() const
  {
    return m_line;
  }

  bool next(std::string& value)
  {
    value.erase();
    std::string s;
    for(m_line++;readNextLine(s);m_line++)
      {
	int pos = 0;
	while(pos < s.length() && (s[pos] == '\t' || s[pos] == ' '))
	  pos++;
	if (pos >= s.length())
	  continue;
	if (s[pos] == '#' || s[pos] == ';')
	  continue;
	while(pos < s.length())
	  value += s[pos++];
	return 1;
      }
    return 0;
    return 0;
  }

private:
  bool readNextLine(std::string& value)
  {
    char c;
    value.erase();
    while(m_stream.get(c))
      {
	if (c == 13)
	  continue;
	if (c == 10)
	  return 1;
	value += c;
      }
      return !value.empty();
  }

private:
  std::istream& m_stream;
  int m_line;
}; //class ConfigFileIterator;

class ConfigFileReader
{
public:
  ConfigFileReader(const std::string& fileName, ConfigFileIterator& iterator): m_fileName(fileName), m_iterator(iterator) {}

  bool read(std::string& deviceName, EventCommandList& commands)
  {
    std::string s;
    if (!m_iterator.next(s))
      {
	printError("device is not specified in your configuration file, it must be the first line");
	return 0;
      }
    assert(!trim(s).empty());
    deviceName = trim(s);
    while(m_iterator.next(s))
      {
	assert(!trim(s).empty());
	int i;
	std::string ss;
	std::vector<std::string> items;
	for(i = 0;i < s.length();i++)
	  {
	    if (items.size() < 3 && s[i] == ':')
	      {
		items.push_back(ss);
		ss.erase();
		continue;
	      } //colon;
	    ss += s[i];
	  } //for();
	items.push_back(ss);
	assert(items.size() <= 4);
	if (items.size() < 4)
	  {
	    printErrorWithLine(m_iterator.getLine(), "too few items separated with \':\', must be four");
	    return 0;
	  }
	int type, code, value;
	std::string command;
	if (trim(items[3]).empty())
	  {
	    printErrorWithLine(m_iterator.getLine(), "command is not specified");
	    return 0;
	  }
	command = trim(items[3]);
	if (!trim(items[0]).empty() && !checkInt(trim(items[0])))
	  {
	    printErrorWithLine(m_iterator.getLine(), "\'" + trim(items[0]) + "\' is not a valid unsigned integer number");
	    return 0;
	  }
	if (trim(items[0]).empty())
	  type = -1; else
	  type = parseInt(trim(items[0]));
	if (!trim(items[1]).empty() && !checkInt(trim(items[1])))
	  {
	    printErrorWithLine(m_iterator.getLine(), "\'" + trim(items[1]) + "\' is not a valid unsigned integer number");
	    return 0;
	  }
	if (trim(items[1]).empty())
	  code = -1; else
	  code = parseInt(trim(items[1])); 
	if (!trim(items[2]).empty() && !checkInt(trim(items[2])))
	  {
	    printErrorWithLine(m_iterator.getLine(), "\'" + trim(items[2]) + "\' is not a valid unsigned integer number");
	    return 0;
	  }
	if (trim(items[2]).empty())
	  value = -1; else
	  value = parseInt(trim(items[2]));
	commands.push_back(EventCommand(command, type, code, value));
      } //while(lines);
    return 1;
  }

private:
  void printError(const std::string& message) const
  {
    std::cerr << ERROR_PREFIX << m_fileName << ":" << message << std::endl;
  }

  void printErrorWithLine(int line, const std::string& message)
  {
    assert(line > 0);
    std::cerr << ERROR_PREFIX << m_fileName << ":" << line << ":" << message << std::endl;
  }

private:
  std::string m_fileName;
  ConfigFileIterator& m_iterator;
}; //class ConfigFileReader;

void runShellCommand(const std::string& command)
{
  assert(!command.empty());
  pid_t pid = fork();
  if (pid == (pid_t)-1)
    throw SystemException("fork()");
  if (pid == (pid_t)0)
    {
      execlp(SHELL, SHELL, "-c", command.c_str(), NULL);
      exit(1);//if execlp() fails;
    }
  waitpid(pid, NULL, 0);
}

int listEvents(const std::string& deviceName)
{
  assert(!deviceName.empty());
  InputEventReader reader(deviceName);
  try 
    {
      reader.open();
    }
  catch(const SystemException& e)
    {
      std::cerr << ERROR_PREFIX << "could not open device" << std::endl;
      std::cerr << e.getComment() << ":" << getErrorMessage(e.getCode()) << std::endl;
      return 1;
    }
  while(1)
    {
      try 
	{
	  InputEvent event = reader.readNextEvent();
	  std::cout << event.type << ":" << event.code << ":" << event.value << std::endl;
	}
      catch(const SystemException& e)
	{
	  std::cerr << ERROR_PREFIX << "could not read next event" << std::endl;
	  std::cerr << e.getComment() << ":" << getErrorMessage(e.getCode()) << std::endl;
	  return 1;
	}
    } //while(1);
  return 0;
}

int mainLoop(InputEventReader& reader, const EventCommandList& commands)
{
  InputEvent event;
  while(1)
    {
      try 
	{
	  event = reader.readNextEvent();
	}
      catch(const SystemException& e)
	{
	  logMessage(LOG_ERROR, e.getComment() + ":" + getErrorMessage(e.getCode()));
	  return 1;
	}
      EventCommandList::const_iterator it;
      for(it = commands.begin();it != commands.end();it++)
	{
	  const EventCommand& c = *it;
	  if (c.getType() != -1 && c.getType() != event.type)
	    continue;
	  if (c.getCode() != -1 && c.getCode() != event.code)
	    continue;
	  if (c.getValue() != -1 && c.getValue() != event.value)
	    continue;
	  break;
	} //for(commands);
      if (it == commands.end())
	continue;
      try {
	runShellCommand(it->getCommand());
      }
      catch(const SystemException& e)
	{
	  logMessage(LOG_ERROR, e.getComment() + ":" + getErrorMessage(e.getCode()));
	}
    } //while(1);
  return 0;
}

void printLogo()
{
  std::cout << "Tagran is a daemon to handle input events. Version: " << VERSION << "." << std::endl;
  std::cout << COPYRIGHT << std::endl;
  std::cout << std::endl;
}

void printHelp()
{
  std::cout << "Usage: tagran [OPTIONS]" << std::endl;
  std::cout << std::endl;
  std::cout << "Options:" << std::endl;
  std::cout << "\t--help, -h - print this help screen;" << std::endl;
  std::cout << "\t--daemon PIDFILE, -d PIDFILE - run in daemon mode and save pid into PIDFILE;" << std::endl;
  std::cout << "\t--list DEVNAME, -l DEVNAME - read events from DEVNAME and print them to stdout" << std::endl;
  std::cout << "\t--config FILENAME, -c FILENAME - read configuration from FILENAME (default " << DEFAULT_CONFIG << ")." << std::endl;
}

int main(int argc, char* argv[])
{
  //FIXME:locale ;
  std::string configFileName = DEFAULT_CONFIG, configPidFileName;
  int i;
  for(i = 1;i < argc;i++)
    {
      std::string value = argv[i];
      if (value == "--help" || value == "-h")
	{
	  printLogo();
	  printHelp();
	  return 0;
	} //help;
      if (value == "--list" || value == "-l")
	{
	  printLogo();
	  i++;
	  if (i >= argc)
	    {
	      std::cerr << ERROR_PREFIX << "device file name is not specified" << std::endl;
	      return 1;
	    } //no device file name;
	  return listEvents(argv[i]);
	} //listEvents;
      if (value == "--daemon" || value == "-d")
	{
	  i++;
	  if (i >= argc)
	    {
	      printLogo();
	      std::cerr << ERROR_PREFIX << "pid file name is not specified" << std::endl;
	      return 1;
	    } //no pid file name;
	  configPidFileName = argv[i];
	  if (trim(configPidFileName).empty())
	    {
	      printLogo();
	      std::cerr << ERROR_PREFIX << "pid file name could not be empty" << std::endl;
		return 1;
	    }
	  continue;
	} //daemonj;
      if (value == "--config" || value == "-c")
	{
	  i++;
	  if (i >= argc)
	    {
	      printLogo();
	      std::cerr << ERROR_PREFIX << "config file name is not specified" << std::endl;
	      return 1;
	    }
	  configFileName = argv[i];
	  if (trim(configFileName).empty())
	    {
	      printLogo();
	      std::cerr << ERROR_PREFIX << "config file name could not be empty" << std::endl;
		return 1;
	    }
	  continue;
	}
      printLogo();
      std::cerr << ERROR_PREFIX << "unknown command line argument \'" << argv[i] << "\'" << std::endl;
      return 1;
    } //for args;
  EventCommandList commands;
  std::ifstream config(configFileName.c_str());
  if (!config)
    {
      printLogo();
      std::cerr << ERROR_PREFIX << "could not open config file for reading (" << configFileName << ")" << std::endl;
      return 1;
    }
  ConfigFileIterator configIterator(config);
  ConfigFileReader configReader(trim(configFileName), configIterator);
  std::string deviceName;
  if (trim(configPidFileName).empty())//in daemon mode;
    printLogo();
  if (!configReader.read(deviceName, commands))
    return 1;
  InputEventReader reader(deviceName);
  try 
    {
      reader.open();
    }
  catch(const SystemException& e)
    {
      std::cerr << ERROR_PREFIX << "could not open input device" << std::endl;
      std::cerr << e.getComment() << ":" << getErrorMessage(e.getCode()) << std::endl;
      return 1;
    }
  if (!trim(configPidFileName).empty())//in daemon mode;
    {
      pid_t pid = fork();
      if (pid == (pid_t)-1)
	{
	  std::cerr << ERROR_PREFIX << "could not create child process" << std::endl;
	    std::cerr << "fork():" << getErrorMessage(errno) << std::endl;
	    return 1;
	} //no child;
      if (pid == (pid_t)0)
	{
	  exit(mainLoop(reader, commands));
	} //child;
      std::ofstream pidFile(configPidFileName.c_str());
      if (pidFile)
	pidFile << pid << std::endl;
      return 0;
    } //daemon;
  return mainLoop(reader, commands);
}
