
#ifndef __VOICEMAN_MAIN_LOOP_H__
#define __VOICEMAN_MAIN_LOOP_H__

#include "ClientFactory.h"
#include"AbstractExecutorOutput.h"

/**\brief The abstract interface for system signal processing classes
 *
 * This class declares method being called in cases when pselect() system
 * call is interrupted by system signal. Usual signal handlers must
 * contain only registration which signal was caught but real signal
 * processing must be implemented in the derived classes of this one. 
 */
class AbstractSignalHandler
{
public:
  virtual ~AbstractSignalHandler() {}

  /**\brief Notifies there was system signal and it must be handled
   *
   * This method is called by main loop class each time when pselect()
   * system call is interrupted with EINTR exit code. It means the process
   * have received system signal and it must be handled.
   */
  virtual void onSystemSignal() = 0;
}; //class AbstractSignalHandler;

/**\brief The abstract class to handle client data
 *
 * This class declares an interface for objects to process data received
 * from clients. The data sent through this interface is not prepared and
 * is not checked. It is provided from MainLoop class. The strings even
 * must not be bounded to line ends.
 *
 * \sa MainLoop ClientDataHandler
 */
class AbstractClientDataHandler
{
public:
  /**\brief Callback method to notify new data was received
   *
   * This class notifies there is new data received from client and it must
   * be handled. The provided string is the raw data block, it can be not
   * bounded to line end, so handler must wait next one until it will be
   * sure the complete string is received.
   *
   * \param [in] client The client object data was received from
   * \param [in] data The received data
   */
  virtual void processClientData(Client& client, const std::string& data) = 0;
}; //class AbstractClientDataHandler;

/**\brief Main class to manage client connections
 *
 * This class manages clients connections and handles any data received
 * from clients. It contains main select() system call to wait new data
 * or new connections. This class has list of currently accepted clients
 * and all clients will be closed on this classs destruction. It is not
 * recommended to have two instances of this class because of behavior
 * may depend on process signal handling.
 */
class MainLoop
{
public:
  /**\brief The constructor
   *
   * \param [in] clientFactory The reference to a factory for new client object creation
   * \param [in] clients The list to store connected client set
   * \param [in] maxClients The maximum number of accepted clients (0 - not limited)
   * \param [in] clientDataHandler The reference to a object to handle data read from client connection
   * \param [in] signalHandler The reference to a object to handle registered signals
   * \param [in] executorOutput The reference to an object to provide information about executor output streams
   * \param [in] terminationFalg The reference to termination flag variable
   */
  MainLoop(const ClientFactory& clientFactory, ClientList& clients, size_t maxClients, AbstractClientDataHandler& clientDataHandler, AbstractSignalHandler& signalHandler, AbstractExecutorOutput& executorOutput, bool& terminationFlag)
    : m_clientFactory(clientFactory), m_connectedClients(clients), m_maxClients(maxClients), m_clientDataHandler(clientDataHandler), m_signalHandler(signalHandler), m_executorOutput(executorOutput), m_terminationFlag(terminationFlag) {}

  /**\brief The main method to execute loop and handle data
   *
   * \param [in] sockets The list of sockets objects to listen
   * \param [in] sigMask The sigmal mask to operate with
   */
  void run(const SocketList& sockets, sigset_t* sigMask);

private:
  const ClientFactory& m_clientFactory;
  ClientList& m_connectedClients;
  const size_t m_maxClients;
  AbstractClientDataHandler& m_clientDataHandler;
  AbstractSignalHandler& m_signalHandler;
  AbstractExecutorOutput& m_executorOutput;
  bool& m_terminationFlag;
}; //class MainLoop;

#endif //__VOICEMAN_MAIN_LOOP_H__
