/*
	Copyright (c) 2000-2009 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the Lopsus website generator.

   Lopsus website generator is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.

   Lopsus website generator is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
*/

#include"lopsus.h"
#include"ConfigFile.h"

#define COMMENT_CHAR(x) ((x) == _T('#'))

class ConfigFileIterator
{
public:
  ConfigFileIterator(std::istream& stream)
    : m_stream(stream), m_line(0) {}

  size_t getLineNumber() const
  {
    return m_line;
  }

  bool next(String& value)
  {
    value.erase();
    std::string ss;
    for(m_line++;readNextLine(ss);m_line++)
      {
	String s = IO2String(ss);
	s = trim(s);
	if (s.empty())
	  continue;
	if (COMMENT_CHAR(s[0]))
	  continue;
	for(std::string::size_type pos = 0;pos < s.length();pos++)
	  {
	    if (COMMENT_CHAR(s[pos]))
	      break;
	    if (s[pos] == _T('\\'))
	      {
		if (pos + 1 >= s.length())
		  value += '\\'; else
		  value += s[pos + 1];
		pos++;
		continue;
	      } //backslash;
	  value += s[pos];
	  } //for();
	return 1;
      } //while();
    return 0;
  }

private:
  bool readNextLine(std::string& value)
  {
    char c;
    value.erase();
    while(m_stream.get(c))
      {
	if (c == '\r')
	  continue;
	if (c == '\n')
	  return 1;
	value += c;
      }
      return !value.empty();
  }

private:
  std::istream& m_stream;
  size_t m_line;
}; //class ConfigFileIterator;

static bool splitLine(const String& line, String& paramName, String& paramValue)
{
  paramName.erase();
  paramValue.erase();
  bool wasEquals = 0;
  for(String::size_type i = 0;i < line.length();i++)
    {
      if (!wasEquals && line[i] == _T('='))
	{
	  wasEquals = 1;
	  continue;
	}  //equals;
      if (wasEquals)
	{
	  paramValue += line[i];
	  continue;
	}
      if (BLANK_CHAR(line[i]))
	continue;
      if ((line[i] >= _T('a') && line[i] <= _T('z')) || (line[i] >= _T('A') && line[i] <= _T('Z')) || DIGIT_CHAR(line[i]))
	{
	  paramName += line[i];
	  continue;
	}
      return 0;
    } //for();
  paramName = trim(toLower(paramName));
  paramValue = trim (paramValue);
  return wasEquals;
}

static bool processUintValue(const String& fileName, size_t line, const String& strValue, size_t& value, String& errorMessage)
{
  if (!checkTypeUnsignedInt(strValue))
    {
      OStringStream ss;
      ss << fileName << _T("(") << line << _T(")\'") << strValue << _T("\' is not a valid unsigned integer value");
      errorMessage = ss.str();
      return 0;
    }
  value = parseAsUnsignedInt(strValue);
  return 1;
}

bool ConfigFile::parse(const String& fileName, String& errorMessage)
{
  std::ifstream is(String2IO(fileName).c_str());
  if (!is)
    {
      errorMessage = fileName + _T(":file is not accessible for reading");
      return 0;
    }
  ConfigFileIterator it(is);
  String line;
  while(it.next(line))
    {
      //iterator skips empty lines;
      String paramName, paramValue;
      if (!splitLine(line, paramName, paramValue))
	{
	  OStringStream ss;
	  ss << fileName << _T("(") << it.getLineNumber() << _T("):invalid configuration file line format");
	  errorMessage = ss.str();
	  return 0;
	}
      if (paramName == _T("pagetitleprefix"))
	pageTitlePrefix = paramValue; else
      if (paramName == _T("pagecontentdirectory"))
	pagesDir = paramValue; else
      if (paramName == _T("outputdirectory"))
	outputDir = paramValue; else
      if (paramName == _T("headerfile"))
	headerFile = paramValue; else
      if (paramName == _T("footerfile"))
	footerFile = paramValue; else
      if (paramName == _T("menufile"))
	menuFile = paramValue; else
      if (paramName == _T("addtohtmlhead"))
	addToHtmlHead = paramValue; else
      if (paramName == _T("newssourcedirectory"))
	newsSourceDir = paramValue; else
      if (paramName == _T("newsdirectory"))
	newsDir = paramValue; else
      if (paramName == _T("lastnewscount"))
	{
	  if (!processUintValue(fileName, it.getLineNumber(), paramValue, lastNewsCount, errorMessage))
	    return 0;
	} else
      if (paramName == _T("indentbeforetitle"))
	{
	  if (!processUintValue(fileName, it.getLineNumber(), paramValue, indentBeforeTitle, errorMessage))
	    return 0;
	} else
      if (paramName == _T("indenttitle"))
	{
	  if (!processUintValue(fileName, it.getLineNumber(), paramValue, indentTitle, errorMessage))
	    return 0;
	} else
      if (paramName == _T("indentbeforeheader"))
	{
	  if (!processUintValue(fileName, it.getLineNumber(), paramValue, indentBeforeHeader, errorMessage))
	    return 0;
	} else
      if (paramName == _T("indentheader"))
	{
	  if (!processUintValue(fileName, it.getLineNumber(), paramValue, indentHeader, errorMessage))
	    return 0;
	} else
      if (paramName == _T("indentbeforemenu"))
	{
	  if (!processUintValue(fileName, it.getLineNumber(), paramValue, indentBeforeMenu, errorMessage))
	    return 0;
	} else
      if (paramName == _T("indentmenu"))
	{
	  if (!processUintValue(fileName, it.getLineNumber(), paramValue, indentMenu, errorMessage))
	    return 0;
	} else
      if (paramName == _T("indentbeforecontent"))
	{
	  if (!processUintValue(fileName, it.getLineNumber(), paramValue, indentBeforeContent, errorMessage))
	    return 0;
	} else
      if (paramName == _T("indentcontent"))
	{
	  if (!processUintValue(fileName, it.getLineNumber(), paramValue, indentContent, errorMessage))
	    return 0;
	} else
      if (paramName == _T("indentbeforefooter"))
	{
	  if (!processUintValue(fileName, it.getLineNumber(), paramValue, indentBeforeFooter, errorMessage))
	    return 0;
	} else
      if (paramName == _T("indentfooter"))
	{
	  if (!processUintValue(fileName, it.getLineNumber(), paramValue, indentFooter, errorMessage))
	    return 0;
	} else
      if (paramName == _T("indentfinal"))
	{
	  if (!processUintValue(fileName, it.getLineNumber(), paramValue, indentFinal, errorMessage))
	    return 0;
	} else
	//add new params here;
	{
	  OStringStream ss;
	  ss << fileName << _T("(") << it.getLineNumber() << _T(")specified parameter name is unknown");
	  errorMessage = ss.str();
	  return 0;
	}
    } //while(lines);
  return 1;
}

void ConfigFile::print()
{
  std::cout << "page title prefix = " << pageTitlePrefix << std::endl;
  std::cout << "page content directory = " << pagesDir << std::endl;
  std::cout << "output directory = " << outputDir << std::endl;
  std::cout << "header file = " << headerFile << std::endl;
  std::cout << "footer file = " << footerFile << std::endl;
  std::cout << "menu file = " << menuFile << std::endl;
  std::cout << "add to html head = " << addToHtmlHead << std::endl;
  std::cout << "news source directory = " << newsSourceDir << std::endl;
  std::cout << "news directory = " << newsDir << std::endl;
  std::cout << "last news count = " << lastNewsCount << std::endl;
  std::cout << "indent before title = " << indentBeforeTitle << std::endl;
  std::cout << "indent title = " << indentTitle << std::endl;
  std::cout << "indent before header = " << indentBeforeHeader << std::endl;
  std::cout << "indent header = " << indentHeader << std::endl;
  std::cout << "indent before menu = " << indentBeforeMenu << std::endl;
  std::cout << "indent menu = " << indentMenu << std::endl;
  std::cout << "indent before content = " << indentBeforeContent << std::endl;
  std::cout << "indent content = " << indentContent << std::endl;
  std::cout << "indent before footer = " << indentBeforeFooter << std::endl;
  std::cout << "indent footer = " << indentFooter << std::endl;
  std::cout << "indent final = " << indentFinal << std::endl;
}
