/*
	Copyright (c) 2000-2009 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the Lopsus website generator.

   Lopsus website generator is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.

   Lopsus website generator is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
*/

#include "lopsus.h"
#include"LopsusPage.h"
#include"LopsusText.h"

#define INDENT_STEP 2

static void translateText(const String& text, StringList& output)
{
  DefaultCommandProcessor commandProcessor;
  LopsusText lopsusText(text, output, commandProcessor);
  lopsusText.translate();
  if (!lopsusText.directives.empty())
    lopsusWarning("Paragraph command used in illegal place");
}

static void translateText(const String& text, StringList& output, StringList& directives)
{
  DefaultCommandProcessor commandProcessor;
  LopsusText lopsusText(text, output, commandProcessor);
  lopsusText.translate();
  directives = lopsusText.directives;
}

static String translateText(const String& text)
{
  StringList lines;
  translateText(text, lines);
  String s;
  for(StringList::const_iterator it = lines.begin();it != lines.end();it++)
    s += *it + _T(" ");
  return trim(s);
}

static void splitLines(const String& text, StringVector& lines)
{
  StringList lineList;
  String s;
  for(String::size_type i = 0;i < text.length();i++)
    {
      if (text[i] == '\r')
	continue;
      if (text[i] == '\n')
	{
	  lineList.push_back(trim(s));
	  s.erase();
	  continue;
	}
      s += text[i];
    }
  lineList.push_back(trim(s));
  lines.reserve(lineList.size());
  for(StringList::const_iterator it = lineList.begin();it != lineList.end();it++)
    {
      if (it->empty())
	{
	  if (lines.empty() || !lines.back().empty())
	    lines.push_back(String());
	} else 
	lines.push_back(*it);
    }
}

static void removeMarks(PageItem& item)
{
  if (item.type == PageItem::Enum)
    {
      String::size_type i = 0;
      while(i < item.text.length() && item.text[i] == _T('*'))
	i++;
      item.level = i;
      while(i < item.text.length() && BLANK_CHAR(item.text[i]))
	i++;
      String s;
      while(i < item.text.length())
	s += item.text[i++];
      item.text = s;
      return;
    }
  if (item.type == PageItem::OrderedEnum)
    {
      String::size_type i = 0;
      while(i < item.text.length() && item.text[i] == _T('#'))
	i++;
      item.level = i;
      while(i < item.text.length() && BLANK_CHAR(item.text[i]))
	i++;
      String s;
      while(i < item.text.length())
	s += item.text[i++];
      item.text = s;
      return;
    }
  if (item.type == PageItem::Header)
    {
      size_t i1 = 0, i2 = 0;
      bool b = 1;
      for(String::size_type i = 0;i < item.text.length();i++)
	{
	  if (item.text[i] != _T('='))
	    b = 0;
	  if (b)
	    i1++;
	  if (item.text[i] == _T('\\'))
	    {
	      i++;
	      continue;
	    }
	  if (item.text[i] == _T('='))
	    i2++; else 
	    i2 = 0;
	} //for();
      size_t level = std::min(i1, i2);
      item.level = level;
      if (item.text.length() <= level * 2)
	{
	  item.text = _T("");
	  return;
	}
      String s;
      for (String::size_type i = level; i < item.text.length() - level;i++)
	s += item.text[i];
      item.text = s;
      return;
    } //Header;
}

static String putInTags(int itemType, size_t level, const StringList& lines, size_t indent, const String& additionalText)
{
  String openTag, closeTag;
  if (itemType == PageItem::Header)
    {
      OStringStream s1, s2;
      s1 << _T("<h") << level << _T(">");
      s2 << _T("</h") << level << _T(">");
      openTag = s1.str();
      closeTag = s2.str();
    }
  if (itemType == PageItem::Enum || itemType == PageItem::OrderedEnum)
    {
      openTag = _T("<li>");
      closeTag = _T("</li>");
    }
  if (itemType == PageItem::Text)
    {
      openTag = _T("<p>");
      closeTag = _T("</p>");
    }
  assert(!lines.empty());
  if (lines.size() == 1 && additionalText.empty())
    {
      String s;
      while(s.length() < indent)
	s += _T(' ');
      s += openTag;
      s += lines.front();
      s += closeTag;
      s += _T('\n');
      return s;
    }
  String s;
  while(s.length() < indent)
    s += _T(' ');
  s += openTag;
  s+= _T('\n');
  for(StringList::const_iterator it = lines.begin();it != lines.end();it++)
    {
      String ss;
      while (ss.length() < indent + INDENT_STEP)
	ss += _T(' ');
      ss += *it;
      ss += _T('\n');
      s += ss;
    } //for(lines);
  s += additionalText;
  String ss;
  while(ss.length() < indent)
    ss += _T(' ');
  ss += closeTag;
  ss += _T('\n');
  s += ss;
  return s;
}

static String openEnum(size_t indent, bool isOrdered)
{
  String s;
  while(s.length() < indent)
    s += _T(' ');
  if (isOrdered)
    s += _T("<ol>\n"); else
    s += _T("<ul>\n");
  return s;
}

static String closeEnum(size_t indent, bool isOrdered)
{
  String s;
  while(s.length() < indent)
    s += _T(' ');
  if (isOrdered)
    s += _T("</ol>\n"); else
    s += _T("</ul>\n");
  return s;
}

static String translateList(PageItemList& list, PageItemList::iterator& it, size_t level, size_t indent)
{
  assert(it != list.end());
  assert(it->type == PageItem::Enum || it->type == PageItem::OrderedEnum);
  String s = openEnum(indent, it->type == PageItem::OrderedEnum);
  bool ordered = it->type == PageItem::OrderedEnum;
  if (it->level > level)
    s += translateList(list, it, level + 1, indent += INDENT_STEP);
  while(it != list.end() && (it->type == PageItem::Enum || it->type == PageItem::OrderedEnum) && it->level == level)
    {
      if (ordered && it->type == PageItem::Enum)
	{
	  s += closeEnum(indent, 1);
	  s += openEnum(indent, 0);
	  ordered = 0;
	} else
      if (!ordered && it->type == PageItem::OrderedEnum)
	{
	  s += closeEnum(indent, 0);
	  s += openEnum(indent, 1);
	  ordered = 1;
	}
      int itemType = it->type;
      StringList lines;
      translateText(it->text, lines);
      it++;
      String additionalText;
      if (it != list.end() && (it->type == PageItem::Enum || it->type == PageItem::OrderedEnum) && it->level > level)
	additionalText = translateList(list, it, level + 1, indent + INDENT_STEP);
      s += putInTags(itemType , level, lines, indent + INDENT_STEP, additionalText);
    } //while();
  s += closeEnum(indent, ordered);
  return s;
}

String LopsusPage::translate(const String& source, const NewsVector& news, size_t desiredIndent)
{
  StringVector lines;
  splitLines(source, lines);
  m_wasEmptyLine = 0;
  splitItems(lines);
  for(PageItemList::iterator it = m_pageItems.begin ();it != m_pageItems.end();it++)
    removeMarks(*it);
  size_t currentIndent = desiredIndent;
  String s;
  for(PageItemList::iterator it = m_pageItems.begin();it != m_pageItems.end();it++)
    {
      if (it->type == PageItem::Enum || it->type == PageItem::OrderedEnum)
	{
	  s += translateList(m_pageItems, it, 1, currentIndent);
	  it--;
	  continue;
	} //if enumeration;
      StringList lines, directives;
      translateText(it->text, lines, directives);
      if (it->type == PageItem::Text && lines.empty())
	if (processDirectives(s, directives, currentIndent, news))
	  continue;
      if (it->type == PageItem::Text && !lines.empty () && !directives.empty())
	lopsusWarning("there are paragraph commands in non-empty paragraph, skipping them...");
      s += putInTags(it->type, it->level, lines, currentIndent, _T(""));
    } //for(page items);
  return s;
}

void LopsusPage::splitItems(const StringVector& lines)
{
  for(StringVector::size_type i = 0;i < lines.size();i++)
    {
      const String& line = lines[i];
      if (line.empty())
	{
	  addEmptyLine();
	  continue;
	}
      if (line[0] == _T('=') && line[line.length() - 1] == _T('='))//FIXME:last = can be escape;
	{
	  addHeaderLine(line);
	  continue;
	}
      if (line[0] == _T('*'))
	{
	  addEnumLine(line);
	  continue;
	}
      if (line[0] == _T('#'))
	{
	  addOrderedEnumLine(line);
	  continue;
	}
      addDefLine(line);
    } //for(lines);
}

void LopsusPage::addEmptyLine()
{
  m_wasEmptyLine = 1;
}

void LopsusPage::addHeaderLine(const String& text)
{
  m_pageItems .push_back(PageItem(PageItem::Header, text));
  m_wasEmptyLine = 0;
}

void LopsusPage::addEnumLine(const String& text)
{
  m_pageItems.push_back(PageItem(PageItem::Enum, text));
  m_wasEmptyLine = 0;
}

void LopsusPage::addOrderedEnumLine(const String& text)
{
  m_pageItems.push_back(PageItem(PageItem::OrderedEnum, text));
  m_wasEmptyLine = 0;
}

void LopsusPage::addDefLine(const String& text)
{
  if (m_pageItems.empty())
    {
      m_pageItems.push_back(PageItem(PageItem::Text, text));
      m_wasEmptyLine = 0;
      return;
    }
  const int lastItemType = m_pageItems.back().type;
  if (lastItemType == PageItem::Header)
    {
      m_pageItems.push_back(PageItem(PageItem::Text, text));
      m_wasEmptyLine = 0;
      return;
    } //Header;
  if (lastItemType == PageItem::Enum)
    {
      if (m_wasEmptyLine)
	{
	  m_pageItems.push_back(PageItem(PageItem::Text, text));
	  m_wasEmptyLine = 0;
	  return;
	}
      attachString(m_pageItems.back().text, text);
      return;
    } //Enum;
  if (lastItemType == PageItem::OrderedEnum)
    {
      if (m_wasEmptyLine)
	{
	  m_pageItems.push_back(PageItem(PageItem::Text, text));
	  m_wasEmptyLine = 0;
	  return;
	}
      attachString(m_pageItems.back().text, text);
      return;
    } //OrderedEnum;
  if (lastItemType == PageItem::Text)
    {
      if (m_wasEmptyLine)
	{
	  m_pageItems.push_back(PageItem(PageItem::Text, text));
	  m_wasEmptyLine = 0;
	  return;
	}
      attachString(m_pageItems.back().text, text);
      return;
    } //Text;
  assert(0);
}

String LopsusPage::generateLastNews(size_t indent, const NewsVector& news) const
{
  const NewsVector::size_type lastIndex = news.size() < m_lastNewsMaxCount?news.size():m_lastNewsMaxCount;
  String s;
  for(NewsVector::size_type i = 0;i < lastIndex;i++)
    s += formatNewsItem(news[i], 1, news[i].relFileName);//1 means with brief content;
  return shiftString(s, indent);
}

String LopsusPage::generateNewsList(size_t indent, const NewsVector& news) const
{
  String s;
  for(NewsVector::size_type i = 0;i < news.size();i++)
    s += formatNewsItem(news[i], 0, news[i].relFileName);//0 means without brief content;
  return shiftString(s, indent);
}

String LopsusPage::formatNewsItem(const NewsItem& newsItem, bool withBriefContent, const String& linkUrl) const
{
  String mainTitle = translateText(newsItem.title);
  if (!linkUrl.empty())
    mainTitle = _T("<a href=\"") + htmlEscapeString(linkUrl) + _T("\">") + mainTitle + _T("</a>");
  mainTitle = _T("<b>") + newsItem.dateTime.dateToHtmlRus() + _T("</b> ") + mainTitle;
  mainTitle = _T("<p>") + mainTitle + _T("</p>");
  String body;
  if (withBriefContent)
    body = _T("<hr>\n");
  body += mainTitle + _T("\n");
  if (withBriefContent && !trim(newsItem.briefContent).empty())
    {
      StringList lines;
      DefaultCommandProcessor commandProcessor;
      LopsusText lopsusText(newsItem.briefContent, lines, commandProcessor);
      lopsusText.translate();
      body += _T("<p>\n");
      for(StringList::const_iterator it = lines.begin();it != lines.end();it++)
	body += _T("  ") + *it + _T("\n");
      body += _T("</p>\n");
    } //if(withBriefContent);
  return body;
}

bool LopsusPage::processDirectives(String& text, const StringList& directives, size_t currentIndent, const NewsVector& news) const
{
  for(StringList::const_iterator strIt = directives.begin();strIt != directives.end();strIt++)
    {
      if (*strIt == _T("lastnews"))
	{
	  text += generateLastNews(currentIndent, news);
	  return 1;
	}
      if (*strIt == _T("newslist"))
	{
	  text += generateNewsList(currentIndent, news);
	  return 1;
	}
      if (*strIt == _T("hr"))
	{
	  text += shiftString<String>(_T("<hr>\n"), currentIndent);
	  return 1;
	}
      lopsusWarning("used unknown paragraph command \'" + String2IO(*strIt) + "\'");
    } //for();
  return 0;
}
