/*
	Copyright (c) 2000-2009 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the Lopsus website generator.

   Lopsus website generator is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.

   Lopsus website generator is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
*/

#include"lopsus.h"
#include"LopsusPage.h"
#include"LopsusText.h"
#include"ConfigFile.h"
#include"ContentFile.h"
#include"news.h"

#define NEWS_FILE_SUFFIX _T(".html")

#define ERROR_PREFIX "lopsus:"
#define WARNING_PREFIX "warning:"
#define CONFIG_FILE _T("config")
#define CONTENT_FILE _T("content")
#define DESIGN_DIR _T("design")

static CMDARG cmdLineParams[] = {
  {'d', "dir", "DIR_NAME", "set processing directory (default \'.\');"},
  {'h', "help", "", "show this help screen."},
  {' ', NULL, NULL, NULL}
};

CmdArgsParser cmdLine(cmdLineParams);
ConfigFile configFile;
ContentFileRecordVector contentRecords;
String processingDir = _T(".");
NewsVector news;
String relativeNewsDir;

String htmlAddToHead;
String htmlBeforeTitle;
String htmlBeforeHeader;
String htmlBeforeMenu;
String htmlBeforeContent;
String htmlBeforeFooter;
String htmlFinal;
String htmlHeader;
String htmlFooter;
String htmlMenu;

String combineUnixPath(const String prefix, const String& path)
{
  assert(!path.empty());
  if (path[0] == _T('/'))//it is absolute path;
    return path;
  return concatUnixPath(prefix, path);
}

String prepareTitle(const String& title)
{
  if (trim(configFile.pageTitlePrefix).empty() && trim(title).empty())
    return _T("NO TITLE");
  if (trim(configFile.pageTitlePrefix).empty())
    return trim(title);
  if (trim(title).empty())
    return htmlEscapeString(trim(configFile.pageTitlePrefix));
  return htmlEscapeString(trim(configFile.pageTitlePrefix)) + _T(" ") + trim(title);
}

String prepareHead(const String& title)
{
  String s;
  s += _T("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" \"http://www.w3.org/TR/html4/loose.dtd\">\n");
  s += _T("<html>\n");
  s += _T("  <head>\n");
  s += _T("    <meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\">\n");
  s += _T("    <title>") + prepareTitle(title) + _T("</title>\n");
  s += _T("    <link href=\"/css/main.css\" rel=\"stylesheet\" type=\"text/css\">\n");
  if (!trim(htmlAddToHead).empty())
    s += shiftString(htmlAddToHead, 4);
  s += _T("  </head>\n");
  return s;
}

//All strings must be in html-prepared form and indented;
void compilePageToFile(const String& content, const String& targetFile, const String& shortTitle, const String& title)
{
  String generatedText = prepareHead(shortTitle);
  generatedText += _T("  <body>\n");
  generatedText += shiftString(htmlBeforeTitle, configFile.indentBeforeTitle);
  generatedText += shiftString(trim(title) + _T("\n"), configFile.indentTitle);
  generatedText += shiftString(htmlBeforeHeader, configFile.indentBeforeHeader);
  generatedText += shiftString(htmlHeader, configFile.indentHeader);
  generatedText += shiftString(htmlBeforeMenu, configFile.indentBeforeMenu);
  generatedText += shiftString(htmlMenu, configFile.indentMenu);
  generatedText += shiftString(htmlBeforeContent, configFile.indentBeforeContent);
  generatedText += content;
  generatedText += shiftString(htmlBeforeFooter, configFile.indentBeforeFooter);
  generatedText += shiftString(htmlFooter, configFile.indentFooter);
  generatedText += shiftString(htmlFinal, configFile.indentFinal);
  generatedText += _T("  </body>\n");
  generatedText += _T("</html>\n");
  ensureDirExists(targetFile);
  saveText(targetFile, encodeUTF8(generatedText));
}

void processPage(const String& sourceFile, const String& targetFile, const String& shortTitle, const String& title)
{
  LopsusPage page(configFile.lastNewsCount);
  String content;
  String sourceText = IO2String(readTextFile(sourceFile));
  if (stringEnds<String>(toLower(sourceFile), _T(".php")))
    content = sourceText; else
    if (stringEnds<String>(toLower(sourceFile), _T(".html")) || stringEnds<String>(toLower(sourceFile), _T(".htm")))
      content += shiftString(sourceText, configFile.indentContent); else
      content += page.translate(sourceText, news, configFile.indentContent);
  compilePageToFile(content, targetFile, htmlEscapeString(shortTitle), title);
}

String readLopsusOrHtml(const String& fileName)
{
  if (fileName.empty())
    return _T("");
  if (stringEnds<String>(toLower(trim(fileName)), _T(".html")) || stringEnds<String>(toLower(trim(fileName)), _T(".htm")))
    return IO2String(readTextFile(fileName));
  LopsusPage page;
  const NewsVector emptyNewsVector;
  String text = IO2String(readTextFile(fileName));
  return page.translate(text, emptyNewsVector);
}

void loadTemplates()
{
  std::cout << "Loading templates..." << std::endl;
  htmlBeforeTitle = IO2String(readTextFile(concatUnixPath<String>(processingDir, _T("template/before_title.html"))));
  htmlBeforeHeader = IO2String(readTextFile(concatUnixPath<String>(processingDir, _T("template/before_header.html"))));
  htmlBeforeMenu = IO2String(readTextFile(concatUnixPath<String>(processingDir, _T("template/before_menu.html"))));
  htmlBeforeContent = IO2String(readTextFile(concatUnixPath<String>(processingDir, _T("template/before_content.html"))));
  htmlBeforeFooter = IO2String(readTextFile(concatUnixPath<String>(processingDir, _T("template/before_footer.html"))));
  htmlFinal = IO2String(readTextFile(concatUnixPath<String>(processingDir, _T("template/final.html"))));
  htmlHeader = readLopsusOrHtml(configFile.headerFile);
  htmlFooter = readLopsusOrHtml(configFile.footerFile);
  htmlMenu = readLopsusOrHtml(configFile.menuFile);
  if (!trim(configFile.addToHtmlHead).empty())
    htmlAddToHead = IO2String(readTextFile(configFile.addToHtmlHead));
}

void processContentRecords()
{
  for(ContentFileRecordVector::size_type i = 0;i < contentRecords.size();i++)
    {
      const String sourceFile = combineUnixPath(configFile.pagesDir, contentRecords[i].sourceFile);
      const String targetFile = combineUnixPath(configFile.outputDir, contentRecords[i].targetFile);
      std::cout << "Generating " << targetFile << "..." << std::endl;
      processPage(sourceFile, targetFile, contentRecords[i].shortTitle, contentRecords[i].title);
    } //for(pages);
}

void processNews()
{
  if (!configFile.newsDir.empty())
    {
      for(NewsVector::size_type i = 0;i < news.size();i++)
	{
	  const NewsVector emptyNewsVector;
	  LopsusPage lopsusPage;
	  String content = lopsusPage.translate(news[i].content, emptyNewsVector, configFile.indentContent);
	  const String fileName = combineUnixPath(configFile.newsDir, news[i].fileName + NEWS_FILE_SUFFIX);
	  DefaultCommandProcessor defaultCommandProcessor;
	  StringList lines;
	  LopsusText lopsusText(news[i].title, lines, defaultCommandProcessor);
	  lopsusText.translate();
	  String title;
	  for (StringList::const_iterator it = lines.begin();it != lines.end();it++)
	    title += *it + _T(" ");
	  String dateAndTitle = _T("<p><b>") + news[i].dateTime.dateToHtmlRus() + _T("</b></p>\n");
	  dateAndTitle += _T("<p>") + title + _T("</p>\n");
	  dateAndTitle += _T("<hr>\n");
	  dateAndTitle = shiftString(dateAndTitle, configFile.indentContent);
	  content = dateAndTitle + content;
	  DefaultCommandProcessor plainCommandProcessor;
	  lines.empty();
	  LopsusText shortTitleText(news[i].title, lines, plainCommandProcessor);
	  shortTitleText.translatePlain();
	  String shortTitle;
	  for(StringList::const_iterator it = lines.begin();it != lines.end();it++)
	    shortTitle += *it + _T(" ");
	  std::cout << "Generating " << fileName << "..." << std::endl;
	  compilePageToFile(content, fileName, shortTitle, title);//shortTitle must be in already escaped form;
	} //for(news);
    } //if();
}

void run()
{
  loadTemplates();
  std::cout << "Copying design files..." << std::endl;
  copyDirContent(concatUnixPath<String>(processingDir, DESIGN_DIR), configFile.outputDir);
  processContentRecords();
  processNews();
}

void adjustPathes()
{
  if (!trim(configFile.newsDir).empty())
    relativeNewsDir = _T("/") + configFile.newsDir;
  configFile.pagesDir = combineUnixPath(processingDir, configFile.pagesDir);
  configFile.outputDir = combineUnixPath(processingDir, configFile.outputDir);
  if (!configFile.headerFile.empty())
    configFile.headerFile = combineUnixPath(processingDir, configFile.headerFile);
  if (!configFile.footerFile.empty())
    configFile.footerFile = combineUnixPath(processingDir, configFile.footerFile);
  if (!configFile.menuFile.empty())
    configFile.menuFile = combineUnixPath(processingDir, configFile.menuFile);
  if (!configFile.addToHtmlHead.empty())
    configFile.addToHtmlHead = combineUnixPath(processingDir, configFile.addToHtmlHead);
  if (!configFile.newsSourceDir.empty())
    configFile.newsSourceDir = combineUnixPath(processingDir, configFile.newsSourceDir);
  if (!configFile.newsDir.empty())
    configFile.newsDir = combineUnixPath(configFile.outputDir, configFile.newsDir);
}

void lopsusWarning(const std::string& message)
{
  std::cerr << WARNING_PREFIX << message << std::endl;
}

int main(int argc, char* argv[])
{
  if (!cmdLine.parse(argc, argv))
    return 1;
  if (cmdLine.used("help"))
    {
      std::cout << "The Lopsus website generator. Version: " << PACKAGE_VERSION << ";" << std::endl;
      std::cout << std::endl;
      std::cout << "Usage: lopsus [OPTIONS]" << std::endl;
      std::cout << "Available options:" << std::endl;
      cmdLine.printHelp();
      return 0;
    }
  if (cmdLine.used("dir"))
    {
      processingDir = IO2String(cmdLine["dir"]);
      if (processingDir.empty())
	{
	  std::cerr << ERROR_PREFIX << "processing directory cannot be empty" << std::endl;
	  return 1;
	}
    }
  String errorMessage;
  std::cout << "Loading configuration..." << std::endl;
  if (!configFile.parse(concatUnixPath<String>(processingDir, CONFIG_FILE), errorMessage))
    {
      std::cerr << errorMessage << std::endl;
      return 1;
    }
  if (!configFile.newsDir.empty() && configFile.newsDir[0] == _T('/'))
    {
      std::cout << ERROR_PREFIX << "news directory specification in config file can not begin with slash" << std::endl;
      return 1;
    }
  adjustPathes();
  configFile.print();
  std::cout << "Reading table of content..." << std::endl;
  if (!readContentFile(concatUnixPath<String>(processingDir, CONTENT_FILE), contentRecords, errorMessage))
    {
      std::cerr << errorMessage << std::endl;
      return 1;
    }
  std::cout << "Read " << contentRecords.size() << " content records;" << std::endl;
  try {
    loadRusMonthesConstants();
    if (!configFile.newsSourceDir.empty())
      {
	std::cout << "Reading news..." << std::endl;
	readNewsList(configFile.newsSourceDir, news);
	std::cout << "Read " << news.size() << " news items;" << std::endl;
	for(NewsVector::size_type i = 0;i < news.size();i++)
	  {
	    news[i].relFileName = news[i].fileName;
	    news[i].relFileName += NEWS_FILE_SUFFIX;
	    news[i].relFileName = combineUnixPath(relativeNewsDir, news[i].relFileName);
	  }
      }
    run();
  }
  catch(const SystemException& e)
    {
      std::cerr << ERROR_PREFIX << e.getMessage() << std::endl;
      return 1;
    }
  catch(const LopsusException& e)
    {
      std::cerr << ERROR_PREFIX << e.getMessage() << std::endl;
      return 1;
    }
  catch (std::bad_alloc)
    {
      std::cerr << ERROR_PREFIX << "no enough free memory" << std::endl;
      return 1;
    }
  return 0;
}
