/*
	Copyright (c) 2000-2009 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the Lopsus website generator.

   Lopsus website generator is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.

   Lopsus website generator is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
*/

#include"lopsus.h"
#include"news.h"

#define LOPSUS_BRIEF_BEGIN _T("_lopsus_brief_content_begin_")
#define LOPSUS_BRIEF_END _T("_lopsus_brief_content_end_")

#define DEFAULT_HOUR 12
#define DEFAULT_MIN 0
#define DEFAULT_SEC 0

static void processDateTimeLine(const String& line, DateTime& res)
{
  String dateStr, timeStr;
  size_t index = 0;
  String::size_type i = 0;
  while (i < line.length())
    {
      while(i < line.length() && BLANK_CHAR(line[i]))
	i++;
      String s;
      while(i < line.length() && !BLANK_CHAR(line[i]))
	{
	  s += line[i];
	  i++;
	}
      if (index == 0)
	dateStr = s;
      if (index == 1)
	timeStr = s;
      index++;
    } //while;
  String dayStr, monthStr, yearStr;
  index = 0;
  for(i = 0;i < dateStr.length();i++)
    {
      if (dateStr[i] == _T('-'))
	{
	  index++;
	  continue;
	}
      switch(index)
	{
	case 0:
	  yearStr += dateStr[i];
	  break;
	case 1:
	  monthStr += dateStr[i];
	  break;
	case 2:
	  dayStr += dateStr[i];
	  break;
	default:
	    LOPSUS_ERROR("extra field in date specification");
	}; //switch(index);
    } //for(dateStr);
  String hourStr, minStr, secStr;
  index = 0;
  for(i = 0;i < timeStr.length();i++)
    {
      if (timeStr[i] == _T(':'))
	{
	  index++;
	  continue;
	}
      switch(index)
	{
	case 0:
	  hourStr += timeStr[i];
	  break;
	case 1:
	  minStr += timeStr[i];
	  break;
	case 2:
	  secStr += timeStr[i];
	  break;
	default:
	    LOPSUS_ERROR("extra field in time specification");
	}; //switch(index);
    } //for (timeStr);
  assert(!dateStr.empty());
  if (yearStr.empty())
    LOPSUS_ERROR("year is not specified");
  if (monthStr.empty())
    LOPSUS_ERROR("month is not specified");
  if (dayStr.empty())
    LOPSUS_ERROR("day is not specified");
  if (!checkTypeUnsignedInt(yearStr))
    LOPSUS_ERROR("invalid year specification");
  if (!checkTypeUnsignedInt(monthStr))
    LOPSUS_ERROR("invalid month specification");
  if (!checkTypeUnsignedInt(dayStr))
    LOPSUS_ERROR("invalid day specification");
  res.year = parseAsUnsignedInt(yearStr);
  res.month = parseAsUnsignedInt(monthStr);
  res.day = parseAsUnsignedInt(dayStr);
  if (res.month < 1 || res.month > 12)
    LOPSUS_ERROR("\'" + String2IO(monthStr) + "\' is not a valid month number");
  if (res.day < 1 || res.day > 31)
    LOPSUS_ERROR("\'" + String2IO(dayStr) + "\' is not a valid day number");
  if (timeStr.empty())
    {
      res.hour = DEFAULT_HOUR;
      res.min = DEFAULT_MIN;
      res.sec = DEFAULT_SEC;
    } else
    {
      if (hourStr.empty())
	LOPSUS_ERROR("hour is not specified");
      if (minStr.empty())
	LOPSUS_ERROR("minute is not specified");
      if(secStr.empty())
	LOPSUS_ERROR("second is not specified");
      if (!checkTypeUnsignedInt(hourStr))
	LOPSUS_ERROR("invalid hour specification");
      if (!checkTypeUnsignedInt(minStr))
	LOPSUS_ERROR("invalid minute specification");
      if(!checkTypeUnsignedInt(secStr))
	LOPSUS_ERROR("invalid second specification");
      res.hour = parseAsUnsignedInt(hourStr);
      res.min = parseAsUnsignedInt(minStr);
      res.sec = parseAsUnsignedInt(secStr);
      if (res.hour > 23)
	LOPSUS_ERROR("\'" + String2IO(hourStr) + "\' is not a valid hour");
      if (res.min > 59)
	LOPSUS_ERROR("\'" + String2IO(minStr) + "\' is not a valid minute");
      if (res.sec > 59)
	LOPSUS_ERROR("\'" + String2IO(secStr) + "\' is not a valid seconds specification");
    }
}

static void processNewsItem(const String& fileName, NewsItem& newsItem)
{
  const String& text = readUTF8(readTextFile(fileName));
  StringList lines;
  splitTextFileLines(text, lines, 0, 0);//0 and 0 means not to skip empty lines and do not trim them;
  StringList briefContentLines, contentLines;
  size_t lineCounter = 0;
  bool processingBriefContent = 0;
  for(StringList::const_iterator it = lines.begin();it != lines.end();it++)
    {
      const String& line = trim(*it);
      if (!line.empty())
	lineCounter++;
      if (!line.empty() && lineCounter == 1)
	{
	  newsItem.title = line;
	  continue;
	} //title;
      if (!line.empty() && lineCounter == 2)
	{
	  processDateTimeLine(line, newsItem.dateTime);
	  continue;
	} //date;
      if (!line.empty() && lineCounter == 3)
	{
	  newsItem.category = line;
	  continue;
	} //date;
      if (toLower(line) == LOPSUS_BRIEF_BEGIN)
	{
	  if (processingBriefContent)
	  lopsusWarning("trying to open brief content block twice");
	  processingBriefContent = 1;
	  continue;
	}
      if (toLower(line) == LOPSUS_BRIEF_END)
	{
	  if (!processingBriefContent)
	    lopsusWarning("trying to close not opened brief content block");
	  processingBriefContent = 0;
	  continue;
	}
      if (processingBriefContent)
	briefContentLines.push_back(*it); else
	contentLines.push_back(*it);
	  } //for(lines);
  newsItem.content.erase();
  newsItem.briefContent.erase();
  for(StringList::const_iterator it = briefContentLines.begin();it != briefContentLines.end();it++)
    newsItem.briefContent += (*it + _T("\n"));
  for(StringList::const_iterator it = contentLines.begin();it != contentLines.end();it++)
    newsItem.content += (*it + _T("\n"));
  if (trim(newsItem.content).empty())
    LOPSUS_ERROR("no content");
}

void readNewsList(const String& fromDir, NewsVector& newsVector)
{
  StringList files;
  getDirContent(fromDir, files, 1);//1 means skip directories;
  newsVector.clear();
  newsVector.reserve(files.size());
  for(StringList::const_iterator it = files.begin();it != files.end();it++)
    {
      const String fileName = concatUnixPath(fromDir, *it);
      NewsItem newsItem;
      newsItem.fileName = *it;
      std::cout << "Reading news item from " << fileName << "..." << std::endl;
      processNewsItem(fileName, newsItem);
      newsVector.push_back(newsItem);
    }
  std::sort(newsVector.begin(), newsVector.end());
}
