/*
	Copyright (c) 2000-2009 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the Lopsus website generator.

   Lopsus website generator is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.

   Lopsus website generator is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
*/

#include"lopsus.h"
#include"LopsusPage.h"

#define ERROR_PREFIX "lopsus-page:"
#define WARNING_PREFIX "warning:"

static CMDARG cmdLineParams[] = {
  {'c', "complete", "TITLE", "generate complete html page with specified title;"},
  {'h', "help", "", "show this help screen;"},
  {'i', "indent", "VALUE", "set indentation value."},
  {' ', NULL, NULL, NULL}
};
CmdArgsParser cmdLine(cmdLineParams);

String prepareHeader(const String& title)
{
  OStringStream s;
  s << "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" \"http://www.w3.org/TR/html4/loose.dtd\">" << std::endl;
  s << "<html>" << std::endl;
  s << "  <head>" << std::endl;
  s << "    <meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\">" << std::endl;
  s << "    <title>" << htmlEscapeString(trim(title)) << "</title>" << std::endl;
  s << "  </head>" << std::endl;
  s << "  <body>" << std::endl;
  return s.str();
}

bool readInputText(String& text)
{
  text.erase();
  if (cmdLine.files.empty() || (cmdLine.files.size() == 1 && trim(cmdLine.files[0]) == "-"))
    {
      std::string inputText;
      char c;
      while(std::cin.get(c))
	inputText += c;
      text = IO2String(inputText);
      return 1;
    } //stdin;
  for (StringVector::size_type i = 0;i < cmdLine.files.size();i++)
    {
      std::string inputText;
      std::ifstream is(cmdLine.files[i].c_str());
      if (!is)
	{
	  std::cerr << ERROR_PREFIX << "\'" << cmdLine.files[i] << "\' is not accessible for reading" << std::endl;
	  return 0;
	} //reading problem;
      char c;
      while(is.get(c))
	inputText += c;
      text += IO2String(inputText);
      text += _T('\n');
    } //for(files);
  return 1;
}

void printHelp()
{
  std::cout << "Lopsus Page: The utility to translate lopsus page content files; Version: " << PACKAGE_VERSION << "." << std::endl;
  std::cout << std::endl;
  std::cout << "\tUsage: lopsus-page [OPTIONS] [FILE1 [FILE2 [...]]]" << std::endl;
  std::cout << std::endl;
  std::cout << "VAlid options are:" << std::endl;
  cmdLine.printHelp();
  std::cout << std::endl;
  std::cout << "If file list is omitted or contains only \'-\' stdin is used for input." << std::endl;
}

void lopsusWarning(const std::string& message)
{
  std::cerr << WARNING_PREFIX << message << std::endl;
}

int main(int argc, char* argv[])
{
  if (!cmdLine.parse(argc, argv))
    return 1;
  if (cmdLine.used("help"))
    {
      printHelp();
      return 0;
    }
  bool completeHtml = 0;
  String pageTitle;
  if (cmdLine.used("complete"))
    {
      completeHtml = 1;
      pageTitle = IO2String(cmdLine["complete"]);
    }
  size_t desiredIndent = 0;
  if (completeHtml)
    desiredIndent = 4;
  if (cmdLine.used("indent"))
    {
      if (!checkTypeUnsignedInt(cmdLine["indent"]))
	{
	  std::cerr << ERROR_PREFIX << "\'" << cmdLine["indent"] << "\' is not a valid indentation value" << std::endl;
	  return 1;
	}
      desiredIndent = parseAsUnsignedInt(cmdLine["indent"]);
    }

  String text;
  if (!readInputText(text))
    return 1;
  try {
    loadRusMonthesConstants();
    String res;
    if (completeHtml)
      res = prepareHeader(pageTitle);
    const NewsVector emptyNewsVector;
    LopsusPage page(10);//10 means ten last news, but it is no matter, because there are no news at all;
    res += page.translate(text, emptyNewsVector, desiredIndent);
    if (completeHtml)
      {
	res += _T("  </body>\n");
	res += _T("</html>\n");
      }
    std::cout << encodeUTF8(res) << std::endl;
  }
  catch(const SystemException& e)
    {
      std::cerr << ERROR_PREFIX << e.getMessage() << std::endl;
      return 1;
    }
  catch (const LopsusException& e)
    {
      std::cerr << ERROR_PREFIX << e.getMessage() << std::endl;
      return 1;
    }
  catch (std::bad_alloc)
    {
      std::cerr << ERROR_PREFIX << "no enough free memory" << std::endl;
      return 1;
    }
  return 0;
}
