/*
	Copyright (c) 2000-2009 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the Lopsus website generator.

   Lopsus website generator is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.

   Lopsus website generator is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
*/

#include"lopsus.h"

#define CREATE_MODE (S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IWGRP | S_IXGRP | S_IROTH | S_IWOTH | S_IXOTH)

static bool isDirectory(const std::string& name)
{
  struct stat st;
  if (stat(name.c_str(), &st) != 0)//object does not exist;
    return 0;
  return S_ISDIR(st.st_mode);
}

static void copyFile(const std::string& fromFile, const std::string& toFile)
{
  int fromFd = open(fromFile.c_str(), O_RDONLY);
  LOPSUS_SYS(fromFd != -1, "open(" + fromFile + ", O_RDONLY)");
  int toFd = creat(toFile.c_str(), CREATE_MODE);
  LOPSUS_SYS(toFd != -1, "creat(" + toFile + ")");
  char buf[2048];
  ssize_t readCount;
  do {
    readCount = read(fromFd, buf, sizeof(buf));
    LOPSUS_SYS(readCount != -1, "read(" + fromFile + ")");
    LOPSUS_SYS(write(toFd, buf, readCount) == readCount, "write(" + toFile + ")");
  } while(readCount > 0);
  close(fromFd);
  close(toFd);
}

std::string readTextFile(const String& fileName)
{
  int fd = open(String2IO(fileName).c_str(), O_RDONLY);
  LOPSUS_SYS(fd!= -1, String2IO(fileName));
  std::string s;
  char buf[2048];
  ssize_t readCount;
  do {
    readCount = read(fd, buf, sizeof(buf));
    LOPSUS_SYS(readCount>=0, "read()");
    for(ssize_t i = 0;i < readCount;i++)
      s += buf[i];
  } while(readCount);
  close(fd);
  return s;
}

void saveText(const String& fileName, const std::string toSave)
{
  int fd = creat(String2IO(fileName).c_str(), CREATE_MODE);
  LOPSUS_SYS(fd != -1, "creat(" + String2IO(fileName) + ")");
  std::string::size_type pos = 0;
  while(pos < toSave.length())
    {
      char buf[2048];
      if (toSave.length() - pos > sizeof(buf))
	{
	  for(std::string::size_type i = 0;i < sizeof(buf);i++)
	    buf[i] = toSave[pos + i];
	  ssize_t wroteCount = write(fd, buf, sizeof(buf));
	  LOPSUS_SYS(wroteCount != -1, "write(" + String2IO(fileName) + ")");
	  pos += wroteCount;
	} else
	{
	  for(std::string::size_type i = 0;i < toSave.length() - pos;i++)
	    buf[i] = toSave[pos + i];
	  ssize_t wroteCount = write(fd, buf, toSave.length() - pos);
	  LOPSUS_SYS(wroteCount != -1, "write(" + String2IO(fileName) + ")");
	  pos += wroteCount;
	}
    } //while();
  close(fd);
}

void ensureDirExists(const String& fileName)
{
  const std::string name = String2IO(fileName);
  std::string s;
  for(std::string::size_type i = 0;i < name.length();i++)
    {
      const std::string ss = s;
      s += name[i];
      if (name[i] != '/')
	continue;
      if (ss.empty())
	continue;
      if (isDirectory(ss + "/."))
	continue;
      LOPSUS_SYS(mkdir(ss.c_str(), CREATE_MODE) == 0, "mkdir(" + s + ")");
    } //for();
}

void copyDirContent(const String& fromDir, const String& toDir)
{
  const std::string from = String2IO(fromDir);
  if (!isDirectory(concatUnixPath<std::string>(from, ".")))
    return;
  DIR* dir = opendir(from.c_str());
  LOPSUS_SYS(dir != NULL, "opendir(" + from + ")");
  struct dirent* ent;
  while((ent = readdir(dir)) != NULL)
    {
      std::string entName(ent->d_name);
      if (entName == "." || entName == "..")
	continue;
      if (isDirectory(concatUnixPath(from, entName)))
	{
	  copyDirContent(concatUnixPath(fromDir, IO2String(entName)), concatUnixPath(toDir, IO2String(entName)));
	  continue;
	}
      struct stat st;
      LOPSUS_SYS(stat(concatUnixPath(from, entName).c_str(), &st) == 0, "stat(" + concatUnixPath(from, entName) + ")");
      if (S_ISREG(st.st_mode))
	{
	  ensureDirExists(concatUnixPath(toDir, IO2String(entName)));
	  copyFile(concatUnixPath(from, entName), concatUnixPath(String2IO(toDir), entName));
	}
    } //while();
  closedir(dir);
}

void getDirContent(const String& dirName, StringList& content, bool noDirs)
{
  const std::string from = String2IO(dirName);
  DIR* dir = opendir(from.c_str());
  LOPSUS_SYS(dir != NULL, "opendir(" + from + ")");
  struct dirent* ent;
  while ((ent = readdir(dir)) != NULL)
    {
      const std::string name(ent->d_name);
      if (name == "." || name == "..")
	continue;
      struct stat s;
      const std::string nameToCheck = concatUnixPath(from, name);
      LOPSUS_SYS(stat(nameToCheck.c_str(), &s) == 0, "stat(" + nameToCheck + ")");
      if (!S_ISREG(s.st_mode) && !S_ISDIR(s.st_mode))
	continue;
      if (noDirs && S_ISDIR(s.st_mode))
	continue;
      content.push_back(IO2String(name));
    }
  closedir(dir);
}

