/*
   Copyright 2000-2012 Michael Pozhidaev

   This file is part of the Textlus.

   Textlus is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.

   Textlus is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
*/

#include"system.h"

#define PACKAGE_VERSION "1.0"

#define PREFIX "textlus:"
#define WARNING_PREFIX "textlus:warning"
#define DEFAULT_REPLACEMENT_FILE_NAME "replacements"

#define IO2String transcoding.trIO2WString
#define String2IO transcoding.trWString2IO
#define encodeUTF8 transcoding.trEncodeUTF8
#define decodeUTF8 transcoding.trDecodeUTF8
#define readUTF8 transcoding.trReadUTF8

#define IO_BUF_SIZE 512
#define BOOKMARK_LINE_SIZE 12

#define ICONV_UTF8_ID "utf8"
#define ICONV_WSTRING_ID "utf32le"
#define ICONV_BLOCK_SIZE 50
#define WSTRING_BAD_CHAR L'?'
#define STRING_BAD_CHAR '?'

#define BLANK_CHAR(x) ((x)==10 || (x)==13 || (x)==9 || (x)==32)

#define TEXTLUS_SYS(expr, msg) if (!(expr)) throw SystemException(msg)

#define SHELL "/bin/sh"

struct CmdArg
{
  char shortName;
  const char* longName;
  const char* param;
  const char*descr;
}; //struct CmdArg;

static CmdArg cmdLineParams[] = {
  {'b', "bookmark", "", "read before and update after reading the bookmark file;"},
  {'h', "help", "", "print this help screen and exit;"},
  {'r', "replacements", "FILENAME", "use replacements from file;"},
  {'s', "synth", "COMMAND", "the command to invoke speech synthesizer."},
  {' ', NULL, NULL, NULL}
};

class SystemException
{
public:
  SystemException(): m_code(errno) {}
  SystemException(const std::string& comment): m_code(errno), m_comment(comment) {}
  SystemException(int code): m_code(code) {}
  SystemException(int code, const std::string& comment): m_code(code), m_comment(comment) {}
  virtual ~SystemException() {}

  int getCode() const
  {
    return m_code;
  }

  std::string getDescr() const
  {
    return strerror(m_code);
  }

  std::string getComment() const
  {
    return m_comment;
  }

  std::string getMessage() const
  {
    return getComment()+":"+getDescr();
  }

private:
  int m_code;
  std::string m_comment;
}; //class SystemException;


class FileTextSource
{
public:
  FileTextSource(const std::string& fileName, const std::string bookmarkFileName)
    : m_fd(-1),
      m_fileName(fileName),
      m_bookmarkFileName(bookmarkFileName),
      m_bookmark(0) {}

  virtual ~FileTextSource() 
  {
    close();
  }

public:
  void open();
  bool moveNext(String& nextPart);
  void close();

private:
  bool getMoreText(std::string& text) const;
  bool getRawBlock(String& s);
  void readBookmarkFile();
  void updateBookmark();

private:
  int m_fd;
  const std::string m_fileName;
  std::string m_bookmarkFileName;
  std::string m_current;
  size_t m_bookmark;
};

class Transcoding //Taken from VoiceMan at 2011-03-30;
{
public:
  Transcoding();
  ~Transcoding() {}

public:
  std::string getIOCharset() const;
  std::wstring trIO2WString(const std::string& s) const;
  std::string trWString2IO(const std::wstring& s) const;
  std::string trEncodeUTF8(const std::wstring& s) const;
  bool trDecodeUTF8(const std::string& s, std::wstring& res) const;
  std::wstring trReadUTF8(const std::string& s) const;

private:
  bool initIConv();
  bool initCurIO();

private:
  std::string m_curIO;
  iconv_t m_iconvIO2WString, m_iconvWString2IO, m_iconvUTF82WString, m_iconvWString2UTF8;
}; //class Transcoding;

std::ostream& operator <<(std::ostream& s, const std::wstring& ws);

class CmdArgsParser//Taken from Musitorius at 2011-04-01;
{
public:
  CmdArgsParser(CmdArg* allParams);
  virtual ~CmdArgsParser() {}

public:
  void printHelp() const;
  bool used(const std::string& name) const;
  const std::string operator [](const std::string& name) const;
  bool parse(int argc, char* argv[]);

public:
  const std::vector<std::string>& files;

private:
  struct USEDPARAM
  {
    USEDPARAM() 
      : hasValue(0) {}

    std::string name, value;
    bool hasValue;
  }; //struct USEDPARAM;

  typedef std::vector<USEDPARAM> UsedParamVector;

private:
  size_t identifyParam(char* p) const;

private:
  typedef std::vector<std::string> StringVector;

  StringVector m_files;
    UsedParamVector m_usedParams;
    CmdArg* m_availableParams;
  size_t m_availableParamCount;
};

class DelimitedFile
{
public:
  typedef std::vector <std::string> StringVector;

  void read(const std::string& fileName);
  size_t getLineCount() const;
  std::string getRawLine(size_t index) const;
  size_t getItemCountInLine(size_t index) const;
  std::string getItem(size_t lineIndex, size_t itemIndex) const;

private:
  StringVector m_lines;
};

static CmdArgsParser cmdLine(cmdLineParams);

static std::string configInputFile;
static std::string configBookmarkFile;
static std::string configSynthCommand;
static std::string configReplacementFile;

static Transcoding transcoding;

void textlusWarning(const std::string& message)
{
  std::cerr << WARNING_PREFIX << message << std::endl;
}

template<class T>
T trim(const T& str)
{
  //FIXME:Optimization;
  typename T::size_type l1=0, l2 = str.length();
  while(l1 < str.length() && BLANK_CHAR(str[l1]))
    l1++;
  while(l2 > l1 && BLANK_CHAR(str[l2-1]))
    l2--;
  T newStr;
  for(typename T::size_type i = l1;i < l2;i++)
    newStr += str[i];
  return newStr;
}

template<class T>
bool checkTypeUnsignedInt(const T& s)
{
  T ss = trim(s);
  if (ss.empty())
    return 0;
  typename T::size_type i = 0;
  if (ss[0] == '+')
    i=1;
  if (i >= ss.length())
    return 0;
  for(;i < ss.length();i++)
    if (ss[i] < '0' || ss[i] > '9')
      return 0;
  return 1;
}

template<class T>
unsigned int parseAsUnsignedInt(const T& s)
{
  T ss = trim(s);
  assert(!ss.empty());
  assert(checkTypeUnsignedInt(ss));
  unsigned int n = 0;
  typename T::size_type i=0;
  if (ss[0] == '+')
    i = 1;
  assert(i<ss.length());
  for(;i < ss.length();i++)
    {
      assert(ss[i] >= '0' || ss[i] <= '9');
      n *= 10;
      n += ss[i] - '0';
    }
  return n;
}

template<class T>
void attachCharWithoutDoubleSpaces(T& str, typename T::value_type ch)
{
  if (!BLANK_CHAR(ch))
    {
      str += ch;
      return;
    }
  if (str.empty())
    return;
  if (BLANK_CHAR(str[str.length() - 1]))
    return;
  str += ' ';
}

bool       onlyDelimiters(const std::string& s, std::string::size_type& pos, const CharSet& delimiters)
{
  while (pos < s.length() && delimiters.find(s[pos]) != delimiters.end())
    pos++;
  return pos >= s.length();
}

void removeLineEndings(String& s)
{
  if (s.length() <= 1)
    return;
  size_t offset = 0;
  for(String::size_type i = 1;i < s.length();i++)
    {
      if (s[i] == '-' && !BLANK_CHAR(s[i - 1]))
	{
	  size_t k = i + 1;
	  while(k < s.length() && s[k] == ' ')
	    k++;
	  if (k + 1 < s.length() && s[k] == '\n')
	    {
	      k++;
	      assert(k > i);
	      offset += (k - i);
	      i = k;
	    }
	}
      assert(i >= offset);
      s[i - offset] = s[i];
    } //for();
  assert(s.length() >= offset);
  s.resize(s.length() - offset);
}

std::string makeBookmarkLine(size_t value)
{
  std::ostringstream ss;
  ss << value;
  std::string s = ss.str();
  while(s.length() < BOOKMARK_LINE_SIZE)
    s += " ";
  return s;
}

std::string readLastLine(int fd)
{
  std::string value, line;
  while(1)
    {
      char buf[IO_BUF_SIZE];
      const ssize_t readCount = ::read(fd, buf, sizeof(buf));
      TEXTLUS_SYS(readCount >= 0, "read()");
      if (readCount == 0)
	return value;
      for (size_t i = 0;i < (size_t)readCount;i++)
	{
	  if (buf[i] == '\n')
	    {
	      line = trim(line);
	      if (!line.empty ())
		value = line;
	      line.erase ();
	      continue;
	    }
	  line += buf[i];
	}
    }
  assert(0);
  return "";//Just to reduce warning messages;
}

void initReplacements(const std::string& replacementsFileName)
{
  /*FIXME:
  m_replacements.clear();
  DelimitedFile file;
  file.read(replacementsFileName);
  for(size_t i = 0;i < file.getLineCount();i++)
    {
      if (file.getItemCountInLine(i) != 2)
	{
	  textlusWarning("Replacements file contains lines with incorrect number of fields");
	  continue;
	}
      String strFrom, strTo;
      strFrom = readUTF8(file.getItem(i, 0));
      strTo = readUTF8(file.getItem(i, 1));
      if (strFrom.empty())
	{
	  textlusWarning("Replacements file contains lines with empty string to replace");
	  continue;
	}
      if (m_replacements.find(strFrom) != m_replacements.end())
	{
	  textlusWarning("Replacements file contains lines with the same string to replace");
	  continue;
	}
      m_replacements.insert(StringToStringMap::value_type(strFrom, strTo));
    }
  */
}

bool findReplacement(const String& str, String::size_type pos, StringToStringMap::const_iterator& result)
{
  /*FIXME:
  for(StringToStringMap::const_iterator it = m_replacements.begin();it != m_replacements.end();++it)
    {
      if (str.length() - pos < it->first.length())
	continue;
      String::size_type j;
      for(j = 0;j < it->first.length();j++)
	if (str[pos + j] != it->first[j])
	    break;
      if (j == it->first.length())
	{
	  result = it;
	  return 1;
	}
    } //for(replacements);
  return 0;
  */
  return 0;
}

String insertReplacements(const String& str)
{
  /*FIXME:
  String newStr;
  for(String::size_type i = 0;i < str.length();i++)
    {
      StringToStringMap::const_iterator it;
      if (findReplacement(str, i, it))
	{
	  assert(it != m_replacements.end());
	  const String& newValue = it->second;
	  if (newValue.length() > 1)
	    {
	      if (BLANK_CHAR(newValue[0]))
		attachSpace(newStr);
	      newStr += trim(newValue);
	      if (BLANK_CHAR(newValue[newValue.length() - 1]))
		attachSpace(newStr);
	    } else 
	    newStr += newValue;
	  i += it->first.length()-1;
	} else
	attachCharWithoutDoubleSpaces(newStr, str[i]);
    }
  return newStr;
  */
  return str;
}

std::string prepareDataFileName(const std::string& fileName)
{
  /*FIXME:
  assert(!fileName.empty());
  if (fileName[0] != '/')
    return concatUnixPath<std::string>(TEXTLUS_DATADIR, fileName);
  return fileName;
  */
  return "";
}

void launchSynth(const String& text)
{
  assert(!configSynthCommand.empty());
  if (trim(text).empty())
    return;
  const std::string toSend = String2IO(text);
  int pp[2];
  TEXTLUS_SYS(pipe(pp) != -1, "pipe()");
  pid_t pid = fork();
  TEXTLUS_SYS (pid != -1, "fork()");
  if (pid == 0)
    {
      int fd = ::open("/dev/null", O_RDONLY);
      if (fd == -1)
	exit(EXIT_FAILURE);
      close(pp[1]);
      dup2(pp[0], STDIN_FILENO);
      dup2(fd, STDOUT_FILENO);
      if (execlp(SHELL, SHELL, "-c", configSynthCommand.c_str(), NULL) == -1)
	exit(EXIT_FAILURE);
    } // child process;
  close(pp[0]);
  const int fd = pp[1];
  TEXTLUS_SYS(write(fd, toSend.c_str(), toSend.length()) != -1, "write()");//There may be short write situation, but our text chunks are quite short, left them as they are;
  TEXTLUS_SYS(write(fd, "\n", 1) != -1, "write()");
  close(fd);
  waitpid(pid, NULL, 0);
}

void run()
{
  FileTextSource fileTextSource(configInputFile, configBookmarkFile);
  //FIXME:  textPreprocessor.init(c.replacementFileName);
  fileTextSource.open();
  String s;
  while (fileTextSource.moveNext(s))
      launchSynth(insertReplacements(s));
  fileTextSource.close();
}

bool loadConfiguration()
{
  if (cmdLine.files.empty())
    {
      std::cerr << PREFIX << "no input files were mentioned in command line" << std::endl;
      return 0;
    }
  if (cmdLine.files.size() > 1)
    {
      std::cerr << PREFIX << "too many input files were listed in the command line" << std::endl;
      return 0;
    }
  configInputFile = cmdLine.files[0];
  if (cmdLine.used("bookmark"))
    configBookmarkFile = configInputFile + ".bookmark";//FIXME:
  if (cmdLine.used("synth"))
    configSynthCommand = cmdLine["synth"];
  if (trim(configSynthCommand).empty())
    {
      std::cerr << PREFIX << "no synth command is provided" << std::endl;
      return 0;
    }
  if (cmdLine.used("replacements"))
    {
      const std::string value = cmdLine["replacements"];
      if (value.empty())
	{
	  std::cerr << PREFIX << "replacement file name cannot be empty" << std::endl;
	  return 0;
	}
      configReplacementFile = prepareDataFileName(value);//FIXME:function name;
    }
  return 1;
}

void printHelp()
{
  std::cout << "Textlus: the utility to read text files with speech synthesizer" << std::endl;
  std::cout << "Version: " << PACKAGE_VERSION << std::endl;
  std::cout << std::endl;
  std::cout << "Usage: textlus [options] [input_file]" << std::endl;
  std::cout << std::endl;
  std::cout << "Valid options are:" << std::endl;
  cmdLine.printHelp();
}

int main(int argc, char* argv[])
{
  if (!cmdLine.parse(argc, argv))
    return 1;
  if (cmdLine.used("help"))
    {
      printHelp();
      return 0;
    }
  if (!loadConfiguration())
    return 1;
  try {
    run();
  }
  catch (const SystemException& e)
    {
      std::cerr << PREFIX << e.getMessage() << std::endl;
      return 1;
    }
  return 0;
}

//FileTextSource;

void FileTextSource::readBookmarkFile()
{
  m_bookmark = 0;
  if (m_bookmarkFileName.empty())
    return;
  const int fd = ::open(m_bookmarkFileName.c_str(), O_RDWR);
  if (fd < 0)
    return;//We silently doing nothing, bookmark file does not exists, but must be created;
  std::string bookmark = readLastLine(fd);
  if (bookmark.empty())//There are no meaningful lines, but can be empty;
    {
      TEXTLUS_SYS(ftruncate(fd, 0) == 0, "ftruncate()");//truncate the file from the current position.
      ::close(fd);
      return;
    }
  if (!checkTypeUnsignedInt(bookmark))
    {
      textlusWarning("File \'" + m_bookmarkFileName + "\' does not contain a valid bookmark, disabling bookmark feature");
      m_bookmarkFileName.erase();
      return;
    }
  m_bookmark = parseAsUnsignedInt(bookmark);
  bookmark = makeBookmarkLine(m_bookmark );
  bookmark += '\n';
  TEXTLUS_SYS(lseek(fd, 0, SEEK_SET) != (off_t)-1, "lseek()");
  TEXTLUS_SYS(write(fd, bookmark.c_str(), bookmark.length()) != (ssize_t)-1, "write()");
  TEXTLUS_SYS(ftruncate(fd, lseek(fd, 0, SEEK_CUR)) == 0, "ftruncate()");//truncate the file from the current position.
  ::close(fd);
}

void FileTextSource::open()
{
  assert(m_fd == -1);
  assert(!m_fileName.empty());
  const int fd = ::open(m_fileName.c_str(), O_RDONLY);
  TEXTLUS_SYS(fd != -1, m_fileName);
  m_fd = fd;
  readBookmarkFile();
  if (m_bookmark > 0)
    TEXTLUS_SYS(lseek(m_fd, m_bookmark, SEEK_SET) != (off_t)-1, "lseek()");
}

bool FileTextSource::moveNext(String& nextPart)
{
  nextPart.erase();
  while(1)
    {
      String s;
      if (!getRawBlock(s))
	return 0;
      s = trim(s);
      if (s.empty())
	continue;
      String::size_type k = 0;
      for(String::size_type i = 0;i < s.length();i++)
	if (s[i] != '\r')
	  s[k++] = s[i];
      s.resize(k);
      removeLineEndings(s);
      for(String::size_type i = 0;i < s.length();i++)
	attachCharWithoutDoubleSpaces(nextPart, s[i]);
      if (trim(nextPart).empty())
	{
	  nextPart.erase();
	  continue;
	}
      return 1;
    }
  assert(0);
  return 0;//Just to reduce warning messages;
}

bool FileTextSource::getRawBlock(String& s)
{
  if (m_current.empty() && !getMoreText(m_current))
    return 0;
  s.erase();
  CharSet delimiters;
  delimiters.insert('.');
  delimiters.insert('!');
  delimiters.insert('?');
  while(1)
    {
      assert(!m_current.empty());
      std::string::size_type delPos = std::string::npos; 
      for(CharSet::const_iterator it = delimiters.begin();it != delimiters.end();it++)
	{
	  const std::string::size_type p = m_current.find(*it, 0);
	  if (p != std::string::npos)
	    if (delPos == std::string::npos || delPos > p)
	      delPos = p;
	}
      if (delPos == std::string::npos)
	{
	  //There is no delimiter, trying to get more text;
	  std::string more;
	  if (!getMoreText(more))
	    {
	      //Nothing more, returning only what we have now;
	      m_bookmark += m_current.length();
	      updateBookmark();
	      s = readUTF8(m_current);
	      m_current.erase();
	      return 1;
	    }
	  m_current += more;
	  continue;
	}
      assert(delPos != std::string::npos && delPos < m_current.length());
      while(onlyDelimiters(m_current, delPos, delimiters))
	{
	  //We have only delimiters at the end of string, but another character required;
	  std::string more;
	  if (!getMoreText(more))
	    {
	      //Nothing more, returning only what we have now;
	      m_bookmark += m_current.length();
	      updateBookmark();
	      s = readUTF8(m_current);
	      m_current.erase();
	      return 1;
	    }
	  m_current += more;
	}
      assert(delPos != std::string::npos && delPos < m_current.length());
      const std::string toReturn = m_current.substr(0, delPos);
      m_bookmark += toReturn.length();
      updateBookmark();
      s = readUTF8(toReturn);
      m_current = m_current.substr(delPos);
      return 1;
    }
  assert(0);
  return 0;//Just to reduce warning messages;
}

bool FileTextSource::getMoreText(std::string& text) const
{
  assert(m_fd != -1);
  text.erase();
  char buf[2048];
  const ssize_t res = ::read(m_fd, buf, sizeof(buf));
  TEXTLUS_SYS(res >= 0, "read(" + m_fileName + ")");
  if (res == 0)
    return 0;
  for(size_t i = 0;i < (size_t)res;i++)
    text += buf[i];
  return 1;
}

void FileTextSource::close()
{
  if (m_fd == -1)
    return;
  ::close(m_fd);
  m_fd = -1;
}

void FileTextSource::updateBookmark()
{
  if (m_bookmarkFileName.empty())
    return;
  std::ofstream os(m_bookmarkFileName.c_str(), std::ios_base::out | std::ios_base::app);
  if (!os)
    {
      //FIXME:show this warning only once;
      textlusWarning("cannot update bookmark in file \'" + m_bookmarkFileName + "\'");
      return;
    }
  os << makeBookmarkLine(m_bookmark) << std::endl;
}

//Transcoding;
//Taken from VoiceMan at 2011-03-30;

Transcoding::Transcoding()
{
  if (!initCurIO())
    {
      std::cerr << "FATAL:There are some problems with selecting charset for I/O operations. Exiting..." << std::endl;
      std::cerr << "Probably environment variable $LANG is not set correctly and contains an invalid value." << std::endl;
      exit(1);
    }
  if (!initIConv())
    {
      std::cerr << "Sorry, there are some problems with preparing iconv library." << std::endl;
      std::cerr << "Please, ensure you have this library support in your system." << std::endl;
      std::cerr << "Exiting..." << std::endl;
      exit(1);
    }
}

std::string Transcoding::getIOCharset() const
{
  return m_curIO;
}

bool Transcoding::initIConv()
{
  m_iconvIO2WString = iconv_open(ICONV_WSTRING_ID, m_curIO.c_str());
  if (m_iconvIO2WString == (iconv_t)-1)
    return 0;
  m_iconvWString2IO = iconv_open(m_curIO.c_str(), ICONV_WSTRING_ID);
  if (m_iconvWString2IO==(iconv_t)-1)
    return 0;
  m_iconvUTF82WString = iconv_open(ICONV_WSTRING_ID, ICONV_UTF8_ID);
  if (m_iconvUTF82WString==(iconv_t)-1)
    return 0;
  m_iconvWString2UTF8 = iconv_open(ICONV_UTF8_ID, ICONV_WSTRING_ID);
  if (m_iconvWString2UTF8==(iconv_t)-1)
    return 0;
  return 1;
}

bool Transcoding::initCurIO()
{
  setlocale(LC_ALL, "");
  std::string lang;
  char* l=getenv("LANG");
  if (!l)
    lang="POSIX"; else
      lang=l;
  ssize_t d=-1;
  std::string::size_type i;
  for(i = 0;i < lang.length();i++)
    if (lang[i]=='.')
      d=i;
  if (d < 0)
    {
      m_curIO="US-ASCII";
      return 1;
    }
  std::string cp;
  for(i = d + 1;i < lang.length();i++)
    cp += lang[i];
  m_curIO = cp;
  return 1;
}

std::ostream& operator <<(std::ostream& s, const std::wstring& ws)
{
  s << String2IO(ws);
  return s;
}

std::wstring Transcoding::trIO2WString(const std::string& s) const
{
  std::wstring res;
  size_t i;
  char* b = new char[s.length()];
  char* bb = b;
  for(i = 0;i < s.length();i++)
    b[i] = s[i];
  std::string::size_type bSize = s.length();
  while(bSize)
    {
      wchar_t* r = new wchar_t[ICONV_BLOCK_SIZE];
      char* rr=(char*)r;
      size_t rsize=ICONV_BLOCK_SIZE*sizeof(wchar_t);
      if (iconv(m_iconvIO2WString, &b, &bSize, &rr, &rsize) == (size_t)(-1))
	{
	  if (errno == EILSEQ)
	    {
	      for(i = 0;i < ICONV_BLOCK_SIZE - rsize / sizeof(wchar_t);i++)
		res += r[i];
	      res += WSTRING_BAD_CHAR;
	      b++;
	      bSize--;
	      delete[] r;
	      continue;
	    } // bad sequence;
	  if (errno == EINVAL)
	    {
	      for(i = 0;i < ICONV_BLOCK_SIZE - rsize / sizeof(wchar_t);i++)
		res += r[i];
	      res += WSTRING_BAD_CHAR;
	      delete[] r;
	      break;
	    }
	  if (errno == E2BIG)
	    {
	      for(i = 0;i < ICONV_BLOCK_SIZE - rsize / sizeof(wchar_t);i++)
		res += r[i];
	      delete[] r;
	      continue;
	    }
	} // (size_t)(-1);
      for(i = 0;i < ICONV_BLOCK_SIZE - rsize / sizeof(wchar_t);i++)
	res += r[i];
      delete[] r;
    } // while;
  delete[] bb;
  return res;
}

std::string Transcoding::trWString2IO(const std::wstring& s) const
{
  std::string res;
  size_t i;
  wchar_t* b = new wchar_t[s.length()];
  char* bb = (char*)b;
  for(i = 0;i < s.length();i++)
    b[i] = s[i];
  std::string::size_type bSize = s.length()*sizeof(wchar_t);
  while(bSize)
    {
      char* r = new char[ICONV_BLOCK_SIZE],* rr = r;
      size_t rsize = ICONV_BLOCK_SIZE;
      if (iconv(m_iconvWString2IO, &bb, &bSize, &rr, &rsize) == (size_t)(-1))
	{
	  if (errno == EILSEQ)
	    {
	      for(i = 0;i < ICONV_BLOCK_SIZE - rsize;i++)
		res += r[i];
	      res += STRING_BAD_CHAR;
	      for(i = 0;i < sizeof(wchar_t);i++)
		bb++;
	      bSize -= sizeof(wchar_t);
	      delete[] r;
	      continue;
	    } // bad sequence;
	  if (errno == EINVAL)
	    {
	      for(i = 0;i < ICONV_BLOCK_SIZE - rsize;i++)
		res += r[i];
	      res += STRING_BAD_CHAR;
	      delete[] r;
	      break;
	    }
	  if (errno == E2BIG)
	    {
	      for( i= 0;i < ICONV_BLOCK_SIZE - rsize;i++)
		res += r[i];
	      delete[] r;
	      continue;
	    }
	} // (size_t)(-1);
      for(i = 0;i < ICONV_BLOCK_SIZE - rsize;i++)
	res += r[i];
      delete[] r;
    } // while;
  delete[] b;
  return res;
}

std::string Transcoding::trEncodeUTF8(const std::wstring& s) const
{
  std::string res;
  size_t i;
  wchar_t* b = new wchar_t[s.length()];
  char* bb = (char*)b;
  for(i = 0;i < s.length();i++)
    b[i] = s[i];
  std::string::size_type bSize = s.length()*sizeof(wchar_t);
  while(bSize)
    {
      char* r = new char[ICONV_BLOCK_SIZE], *rr = r;
      size_t rsize = ICONV_BLOCK_SIZE;
      if (iconv(m_iconvWString2UTF8, &bb, &bSize, &rr, &rsize) == (size_t)(-1))
	{
	  if (errno == EILSEQ)
	    {
	      for(i = 0;i < ICONV_BLOCK_SIZE - rsize;i++)
		res += r[i];
	      res += STRING_BAD_CHAR;
	      for(i = 0;i < sizeof(wchar_t);i++)
		bb++;
	      bSize -= sizeof(wchar_t);
	      delete[] r;
	      continue;
	    } // bad sequence;
	  if (errno == EINVAL)
	    {
	      for(i = 0;i < ICONV_BLOCK_SIZE - rsize;i++)
		res += r[i];
	      res += STRING_BAD_CHAR;
	      delete[] r;
	      break;
	    }
	  if (errno == E2BIG)
	    {
	      for(i = 0;i < ICONV_BLOCK_SIZE - rsize;i++)
		res += r[i];
	      delete[] r;
	      continue;
	    }
	} // (size_t)(-1);
      for(i = 0;i < ICONV_BLOCK_SIZE - rsize;i++)
	res += r[i];
      delete[] r;
    } // while;
  delete[] b;
  return res;
}

bool Transcoding::trDecodeUTF8(const std::string& s, std::wstring& res) const
{
  res.erase();
  size_t i;
  char* b = new char[s.length()];
  char* bb = b;
  for(i = 0;i < s.length();i++)
    b[i] = s[i];
  std::string::size_type bSize = s.length();
  while(bSize)
    {
      wchar_t* r = new wchar_t[ICONV_BLOCK_SIZE];
      char* rr = (char*)r;
      size_t rsize = ICONV_BLOCK_SIZE*sizeof(wchar_t);
      if (iconv(m_iconvUTF82WString, &b, &bSize, &rr, &rsize) == (size_t)(-1))
	{
	  if (errno == EILSEQ)
	    {
	      delete[] r;
	      delete[] bb;
	      return 0;
	    } // bad sequence;
	  if (errno == EINVAL)
	    {
	      delete[] r;
	      delete[] bb;
	      break;
	    }
	  if (errno == E2BIG)
	    {
	      for(i = 0;i < ICONV_BLOCK_SIZE - rsize / sizeof(wchar_t);i++)
		res += r[i];
	      delete[] r;
	      continue;
	    }
	} // (size_t)(-1);
      for(i = 0;i < ICONV_BLOCK_SIZE - rsize / sizeof(wchar_t);i++)
	res += r[i];
      delete[] r;
    } // while;
  delete[] bb;
  return 1;
}

std::wstring Transcoding::trReadUTF8(const std::string& s) const
{
  std::wstring res;
  size_t i;
  char* b = new char[s.length()];
  char* bb = b;
  for(i = 0;i < s.length();i++)
    b[i] = s[i];
  std::string::size_type bSize = s.length();
  while(bSize)
    {
      wchar_t* r = new wchar_t[ICONV_BLOCK_SIZE];
      char* rr = (char*)r;
      size_t rsize = ICONV_BLOCK_SIZE*sizeof(wchar_t);
      if (iconv(m_iconvUTF82WString, &b, &bSize, &rr, &rsize) == (size_t)(-1))
	{
	  if (errno == EILSEQ)
	    {
	      for(i = 0;i < ICONV_BLOCK_SIZE - rsize / sizeof(wchar_t);i++)
		res += r[i];
	      res += WSTRING_BAD_CHAR;
	      res += WSTRING_BAD_CHAR;
	      res += WSTRING_BAD_CHAR;
	      delete[] r;
	      break;
	    } // bad sequence;
	  if (errno == EINVAL)
	    {
	      for(i = 0;i < ICONV_BLOCK_SIZE - rsize / sizeof(wchar_t);i++)
		res += r[i];
	      res += WSTRING_BAD_CHAR;
	      delete[] r;
	      break;
	    }
	  if (errno == E2BIG)
	    {
	      for(i = 0;i < ICONV_BLOCK_SIZE - rsize / sizeof(wchar_t);i++)
		res += r[i];
	      delete[] r;
	      continue;
	    }
	} // (size_t)(-1);
      for(i = 0;i < ICONV_BLOCK_SIZE - rsize / sizeof(wchar_t);i++)
	res += r[i];
      delete[] r;
    } // while;
  delete[] bb;
  return res;
}

//CmdArgsParser;

//Taken from Musitorius at 2011-04-01;

CmdArgsParser::CmdArgsParser(CmdArg* allParams)
  :files(m_files), m_availableParams(allParams), m_availableParamCount(0)
{
  while(m_availableParams[m_availableParamCount].shortName != ' ')
    m_availableParamCount++;
}

void CmdArgsParser::printHelp() const
{
  StringVector keys;
  for(size_t i = 0;i < m_availableParamCount;i++)
    {
      std::string s;
      bool hasParam = (m_availableParams[i].param != NULL && m_availableParams[i].param[0] != '\0');
      s = "  -";
      s += m_availableParams[i].shortName;
      if (hasParam)
	{
	  s += " ";
	  s += m_availableParams[i].param;
	}
      s += ", --";
      s += m_availableParams[i].longName;
      if (hasParam)
	{
	  s += " ";
	  s += m_availableParams[i].param;
	}
      keys.push_back(s);
    }
  std::string::size_type l = 0;
  for(StringVector::size_type i = 0;i < keys.size();i++)
    if (l < keys[i].length())
      l = keys[i].length();
  assert(keys.size() == m_availableParamCount);
  for(size_t i = 0;i < m_availableParamCount;i++)
    {
      using namespace std;
      std::string descr = m_availableParams[i].descr;
      cout << keys[i];
      for(std::string::size_type j = keys[i].length();j < l;j++)
	cout << " ";
      cout << " - " << descr << endl;
    }
}

bool CmdArgsParser::used(const std::string& name) const
{
  for(UsedParamVector::size_type i = 0;i < m_usedParams.size();i++)
    if (m_usedParams[i].name == name)
      return 1;
  return 0;
}

const std::string CmdArgsParser::operator[](const std::string& name) const
{
  assert(used(name));
  for(UsedParamVector::size_type i = 0;i < m_usedParams.size();i++)
    {
      if (m_usedParams[i].name != name)
	continue;
      assert(m_usedParams[i].hasValue);
      return m_usedParams[i].value;
    }
  assert(false);
}

/*
 * This method returns:
 * -1 - if it is not a known parameter;
 * -2 - if it is not a valid command line item.
 */
size_t CmdArgsParser::identifyParam(char* p) const
{
  std::string s = p;
  if (s.length() < 2)
    return (size_t)-1;
  if (s[0] != '-')
    return (size_t)-1;
  if (s[1] != '-')
    {
      if (s.length() != 2)
	return (size_t)-1;
      for(size_t i = 0;i < m_availableParamCount;i++)
	if (s[1]  == m_availableParams[i].shortName)
	  return i;
      return (size_t)-2;
    }
  std::string ss;
  for(std::string::size_type i = 2;i < s.length();i++)
    ss += s[i];
  for(size_t i = 0;i < m_availableParamCount;i++)
    {
      std::string longName = m_availableParams[i].longName;
      if (ss == longName)
	return i;
    }
  return -2;
}

bool CmdArgsParser::parse(int argc, char* argv[])
{
  if (argc < 1)
    {
      std::cerr << "error: Too few arguments in argv[] array." << std::endl;
      return 0;
    }
  for(int i = 1;i < argc;i++)
    {
      size_t p = identifyParam(argv[i]);
      if (p == (size_t)-2)
	{
	  std::cerr << "Error:Unknown command line parameter \'" << argv[i] << "\'." << std::endl;
	  return 0;
	}
      if (p == (size_t)-1)
	{
	  m_files.clear();
	  for(int z = i;z < argc;z++)
	    m_files.push_back(argv[z]);
	  return 1;
	} //it was first file name (not a known argument);
      USEDPARAM up;
      up.name = m_availableParams[p].longName;
      if (m_availableParams[p].param[0])
	{
	  if (i + 1 >= argc)
	    {
	      std::cerr << "error: Argument \'" << argv[i] << "\' requires a additional parameter \'" << m_availableParams[p].param << "\', but it is last item in command line. Be careful." << std::endl;
	      return 0;
	    }
	  i++;
	  up.value = argv[i];
	  up.hasValue = 1;
	}
      m_usedParams.push_back(up);
    } //for(argv);
  return 1;
}

//DelimitedFile;
//Taken from VoiceMan at 2011-04-14;

static std::string readTextFile(const std::string& fileName)
{
  const int fd = open(fileName.c_str(), O_RDONLY);
  TEXTLUS_SYS(fd!= -1, fileName);
  std::string s;
  char buf[2048];
  ssize_t readCount;
  do {
    readCount = read(fd, buf, sizeof(buf));//FIXME:incomplete read operation;
    TEXTLUS_SYS(readCount>=0, "read()");
    for(ssize_t i = 0;i < readCount;i++)
      s += buf[i];
  } while(readCount);
  close(fd);
  return s;
}

static std::string cutComment(const std::string& line)
{
  std::string s;
  bool inQuotes = 0;
  for(std::string::size_type i = 0;i < line.length();i++)
    {
      if (!inQuotes && line[i] == '\"')
	{
	  s += '\"';
	  inQuotes = 1;
	  continue;
	}
      if (inQuotes && line[i] == '\"')
	{
	  if (i + 1 < line.length() && line[i + 1] == '\"')
	    {
	      s += "\"\"";
	      i++;
	      continue;
	    }
	  s += '\"';
	  inQuotes = 0;
	  continue;
	}
      if (line[i] == '#')
	{
	  if (!inQuotes)
	    return s;
	  s += '#';
	  continue;
	}
      s += line[i];
    } //for();
  return s;
}

void DelimitedFile::read(const std::string& fileName)
{
  m_lines.clear();
  m_lines.clear();
  std::string s = readTextFile(fileName);
  s += '\n';
  std::string ss;
  for(std::string::size_type i = 0;i < s.length();i++)
    {
      if (s[i] == '\r')
	continue;
      if (s[i] == '\n')
	{
	  ss = cutComment(ss);
	  if (!trim(ss).empty())
	    m_lines.push_back(ss);
	  ss.erase();
	  continue;
	} // if '\n';
      ss+= s[i];
    } // for;
}

size_t DelimitedFile::getItemCountInLine(size_t index) const
{
  assert(index < m_lines.size());
  const std::string& line=m_lines[index];
  size_t k = 0;
  bool inQuotes = 0;
  for(std::string::size_type i = 0;i < line.length();i++)
    {
      if (!inQuotes && line[i] == '\"')
	{
	  inQuotes = 1;
	  continue;
	}
      if (inQuotes && line[i] == '\"')
	{
	  if (i + 1 < line.length() && line[i + 1] == '\"')
	    {
	      i++;
	      continue;
	    }
	  inQuotes = 0;
	  continue;
	}
      if (!inQuotes && line[i] == ':')
	k++;
      assert(inQuotes || line[i] != '#');
    } //for();
  return k + 1;
}

std::string DelimitedFile::getItem(size_t lineIndex, size_t itemIndex) const
{
  assert(lineIndex < m_lines.size());
  const std::string& line = m_lines[lineIndex];
  std::string value;
  size_t k = 0;
  bool inQuotes = 0;
  for(std::string::size_type i = 0;i < line.length();i++)
    {
      assert(line[i] != '\n' && line[i] != '\r');
      if (!inQuotes && line[i] == '\"')
	{
	  inQuotes = 1;
	  continue;
	}
      if (inQuotes && line[i] == '\"')
	{
	  if (i + 1 < line.length() && line[i + 1] == '\"')
	    {
	      value += '\"';
	      i++;
	      continue;
	    }
	  inQuotes = 0;
	  continue;
	}
      if (inQuotes)
	{
	  value += line[i];
	  continue;
	}
      if (line[i] == ':')
	{
	  if (k == itemIndex)
	    return value;
	  value.erase();
	  k++;
	  continue;
	}
      value += line[i];
    } //for(i);
  if (k == itemIndex)
    return value;
  assert(0);
  return "";//just to reduce warning messages;
}

size_t DelimitedFile::getLineCount() const
{
  return m_lines.size();
}

std::string DelimitedFile::getRawLine(size_t index) const
{
  assert(index <= m_lines.size());
  return m_lines[index];
}
