/*
	Copyright (c) 2000-2006 Michael Pozhidaev<msp@altlinux.org>. 
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"
#include"langs.h"
#include"command.h"

vm_string vm_command_player::prepare_value(vm_param value, uint format, float min, float max) const
{
  if (format > 10)
    {
      vmlog << vmerror << WSTR("Format of value is too large. Returning zero.") << vmendl;
      return WSTR("0");
    }
  return vm_string(value.get_value(min, max), format);
}

vm_string vm_command_player::prepare_cmd_line(vm_param v, vm_param p, vm_param r) const
{
  vm_string volume, rate, pitch;
  volume=prepare_value(v, m_volume_format, m_min_volume, m_max_volume);
  pitch=prepare_value(p, m_pitch_format, m_min_pitch, m_max_pitch);
  rate=prepare_value(r, m_rate_format, m_min_rate, m_max_rate);
  vm_string ss;
  uint i;
  for(i=0;i<m_cmd_line.length();i++)
    {
      if (m_cmd_line[i] != WSTR('%') || i+1>=m_cmd_line.length())
	{
	  ss+=m_cmd_line[i];
	  continue;
	}
      i++;
      if (m_cmd_line[i]==WSTR('v'))
	ss += volume; else
	  if (m_cmd_line[i]==WSTR('p'))
	    ss +=pitch; else
	      if (m_cmd_line[i]==WSTR('r'))
		ss += rate; else
		  ss += m_cmd_line[i];
    }
  return ss;
}

void vm_command_player::play(const vm_string &text, vm_param volume, vm_param pitch, vm_param rate, const std::vector<bool> &cap_mask)
{
  vm_string cmd=prepare_cmd_line(volume, pitch, rate);
  int pp[2];
  stop();
  if (m_pg) while(waitpid(-1*m_pg, NULL, 0)>=0);
  VM_SYS(pipe(pp)>=0, "pipe()");
  VM_SYS((m_pg = fork())>=0, "fork()");
  if (m_pg == 0)
    {
      setpgrp();
      close(pp[1]);
      int fd;
      VM_SYS((fd = open(NULL_DEVICE, O_WRONLY))>=0, "open()");
      dup2(pp[0], STDIN_FILENO);
      dup2(fd, STDOUT_FILENO);
      dup2(fd, STDERR_FILENO);
      VM_SYS(execlp("/bin/sh", "/bin/sh", "-c", vm_string2io(cmd).c_str(), NULL) != -1, "execlp()");
    } // new process
  close(pp[0]);
  vm_string t=text;
  if (cap_mask.size() != 0)
    make_caps(t, cap_mask);
  string s = make_utf8(t);
  s += '\n';
  VM_SYS(write(pp[1], s.c_str(), s.length()) != -1, "write()");
  close(pp[1]);
}

void vm_command_player::stop()
{
  if (m_pg == 0)
    return;
  killpg(m_pg, SIGINT);
  killpg(m_pg, SIGKILL);
}

bool vm_command_player::processing() const
{
  if (m_pg == 0)
    return 0;
  pid_t p=waitpid(m_pg, NULL, WNOHANG);
  if (p > 0)
    while(waitpid(-1*m_pg, NULL, 0)>=0);
  return p==0;
}

static void take_value(const vm_string &name, const vm_string &str, uint& value)
{
  if (str.trim().empty())
    return;
  if (!str.is<uint>())
    {
      vmlog << vmwarn << WSTR("Bad parameter in command output \'") << name << WSTR("\'. \'") << str.trim() << WSTR("\' is not a valid number.") << vmendl;
      return;
    }
  value=str.to<uint>();
}

static void take_value_double(const vm_string &name, const vm_string &str, double& value)
{
  if (str.trim().empty())
    return;
  if (!str.is_double())
    {
      vmlog << vmwarn << WSTR("Bad parameter in command output \'") << name << WSTR("\'. \'") << str.trim() << WSTR("\' is not a valid number.") << vmendl;
      return;
    }
  value=str.to_double();
}

bool vm_command_player::init(vm_conf_parser::section &s)
{
  m_name=s[WSTR("name")];
  if (!s.check(WSTR("command")))
    {
      vmlog << vmerror << WSTR("Command output \'") << m_name << WSTR("\' has no \'command\' parameter.") << vmendl;
      return 0;
    }
  m_cmd_line=s[WSTR("command")];
  vm_string volume, pitch, rate;
  if (s.check(WSTR("volume")))
    volume = s[WSTR("volume")];
  if (s.check(WSTR("pitch")))
    pitch = s[WSTR("pitch")];
  if (s.check(WSTR("rate")))
    rate = s[WSTR("rate")];
vm_string
    volume_format(volume.substr(0)), volume_min(volume.substr(1)), volume_max(volume.substr(2)),
    pitch_format(pitch.substr(0)), pitch_min(pitch.substr(1)), pitch_max(pitch.substr(2)),
    rate_format(rate.substr(0)), rate_min(rate.substr(1)), rate_max(rate.substr(2));
  take_value(m_name, volume_format, m_volume_format);
  take_value_double(m_name, volume_min, m_min_volume);
  take_value_double(m_name, volume_max, m_max_volume);
  take_value(m_name, pitch_format, m_pitch_format);
  take_value_double(m_name, pitch_min, m_min_pitch);
  take_value_double(m_name, pitch_max, m_max_pitch);
  take_value(m_name, rate_format, m_rate_format);
  take_value_double(m_name, rate_min, m_min_rate);
  take_value_double(m_name, rate_max, m_max_rate);
  if (s.check(WSTR("caplist")))
    init_cap_map(s[WSTR("caplist")]);
  return 1;
}

void vm_command_player::make_caps(vm_string &text, const std::vector<bool> &cap_mask) const
{
  assert(text.length()==cap_mask.size());
  vm_string s;
  for(int i=0;i<cap_mask.size();i++)
    {
      if (cap_mask[i])
	{
	  vm_string ss;
	  ss+=text[i];
	  ss=lower_case(ss);
	  hash_map<vm_char, vm_string>::const_iterator it=m_cap_map.find(ss[0]);
	  if (it!=m_cap_map.end())
	    s.space_undup_attach(vm_string(WSTR(" "))+it->second+WSTR(" ")); else
	      s.space_undup_attach(vm_string(WSTR(" "))+text[i]+WSTR(" "));
	} else
	  s.space_undup_attach(text[i]);
    }
  text=s.trim();
}

void vm_command_player::init_cap_map(const vm_string& s)
{
  vm_string::iterator it=s.create_iterator(chars());
  while(it.next())
    {
      vm_string s1=lower_case(it.str());
      if (!it.next())
	{
	  vmlog << vmwarn << WSTR("Uncomplete pair in cap values for output \'") << m_name << WSTR("\'.") << vmendl;
	  continue;
	}
      vm_string s2=it.str();
      if (s1.length() != 1)
	vmlog << vmwarn << WSTR("First pair item for cap list \'") << s1 << WSTR("\' should have only one character. Output: \'") << m_name << WSTR("\'.") << vmendl;
      m_cap_map[s1[0]]=s2;
#ifdef VOICEMAN_DEBUG
      vmlog << vminfo << WSTR("Adding cap item: \'") << s2 << WSTR("\'.") << vmendl;
#endif //VOICEMAN_DEBUG
    }
}

vm_string vm_command_player::lower_case(const vm_string& s) const
{
  vm_lang* l=language_mapper[m_name];
  if (l!= NULL)
    return l->low_case(s);
  return s.lower_case();
}

vm_string vm_command_player::upper_case(const vm_string& s) const
{
  vm_lang* l=language_mapper[m_name];
  if (l != NULL)
    return l->up_case(s);
  return s.upper_case();
}

vm_string vm_command_player::chars() const
{
  vm_lang *l=language_mapper[m_name];
  if (l!=NULL)
    return l->get_chars();
  return ENG_LETTERS;
}
