/*
	Copyright (c) 2000-2006 Michael Pozhidaev<msp@altlinux.org>. 
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"
#include"conf.h"
#include"clients.h"
#include"vmdp.h"
#include"proc.h"
#include"tq.h"
#include"tone.h"
#include"dispatcher.h"

int config_maxline=0;
bool config_tones=1, config_tones_in_queue=1;

void vm_dispatcher::text(vm_client::ptr client, const vm_string &t)
{
  m_proc.line(t, client->m_volume, client->m_pitch, client->m_rate);
}

void vm_dispatcher::letter(vm_client::ptr client, vm_char c)
{
  tq_stop();
  m_proc.letter(c, client->m_volume, client->m_pitch, client->m_rate);
}

void vm_dispatcher::stop(vm_client::ptr client)
{
  tq_stop();
}

void vm_dispatcher::tone(vm_client::ptr client, int freq, int lengthms)
{
  if (!config_tones)
    return;
  if (lengthms < 10)
    {
      vmlog << vmerror << WSTR("Could not produce tone shorter than 10ms.") << vmendl;
	return; 
    }
  if (freq < 50 || freq > 5000)
    {
      vmlog << vmerror << WSTR("Could not produce tone with frequency less than 50Hz and greater than 5kHz.") << vmendl;
      return;
}
  if (!config_tones_in_queue)
    {
      vm_tone tone;
      tone.play(freq, lengthms);
    } else
      tq_add_tone(freq, lengthms);
}

void vm_dispatcher::param(vm_client::ptr client, int param_type, const vm_param &p)
{
  switch(param_type)
    {
    case PARAM_VOLUME:
      client->m_volume=p;
      break;
    case PARAM_PITCH:
      client->m_pitch=p;
      break;
    case PARAM_RATE:
      client->m_rate=p;
      break;
    default:
      assert(0);
    }
}

bool vm_dispatcher::init(vm_conf::ptr conf)
{
  int i;
  for(i=0;i<conf->get_chars_table_item_count();i++)
    {
      vm_char c;
      vm_string s;
      conf->get_chars_table_item(i, c, s);
      vm_chars_table[c]=s;
    }

  vm_text_splitter::ptr splitter=new vm_text_splitter();
  vm_text_replacer::ptr replacer=new vm_text_replacer();
  m_proc.set_text_replacer(replacer);
  m_proc.set_text_splitter(splitter);

  for(i=0;i<conf->m_sections.size();i++)
    {
      vm_conf::section &s=conf->m_sections[i];
      if (s.m_name != WSTR("output"))
	continue;
      vm_string name = s[WSTR("name")];
      vm_lang *l=language_mapper[name];
      if (l == NULL)
	vmlog << vmwarn << WSTR("Language mapper returned NULL pointer for output \'") << name << WSTR("\'.") << vmendl; else
	  splitter->set_map_items(name, l->get_chars());
      if (s.check(WSTR("chars")))
	splitter->set_map_items(name, s[WSTR("chars")]);
    } // for outputs 
  splitter->m_default_output=conf->get_default_output();
  if (splitter->m_default_output.empty())
    vmlog << vminfo << WSTR("Default output is not set.") << vmendl; else
      vmlog << vminfo << WSTR("Default output is \'") << conf->get_default_output() << WSTR("\'.") << vmendl;
  splitter->set_map_items(WSTR("default"), conf->get_default_chars());

  for(i=0;i<conf->get_replacement_count();i++)
    {
      vm_string output, old_str, new_str;
      conf->get_replacement(i, output, old_str, new_str);
      replacer->add_replacement(output, old_str, new_str);
    }
  return 1;
}

vm_string vm_dispatcher::split_and_process(const vm_string &data, vm_client::ptr client)
{
  vm_string s;
  int i;
  for(i=0;i<data.length();i++)
    {
      if (data[i]==WSTR('\n'))
	{
	  if (!s.empty() && !client->m_rejecting)
	    line(s, client);
	  s.erase();
	  client->m_rejecting=0;
	  continue;
	}
      if (config_maxline != 0 && s.length() > config_maxline)
	{
	  vmlog << vmwarn << WSTR("Received string exceeds input line length limit. Rejecting.") <<vmendl;
	  s.erase();
	  client->m_rejecting=1;
	}
      if (!client->m_rejecting)
	s+=data[i];
    }
  return s;
}

void vm_dispatcher::handle_data(const vm_string &data, vm_client::ptr client)
{
  vm_string s;
  int i;
  for(i=0;i<data.length();i++)
    if (data[i]!=13) 
      s += data[i];
  client->m_chain+=s;
  client->m_chain=split_and_process(client->m_chain, client);
}
