/*
	Copyright (c) 2000-2006 Michael Pozhidaev<msp@altlinux.org>. 
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/
			 
#include"voiceman.h"
#include"clients.h"
#include"conf.h"
#include"tq.h"
#include"command.h"
#include"proc.h"
#include"vmdp.h"
#include"dispatcher.h"

static vm_string config_pidfile, config_say, config_datadir, config_message;
static bool config_daemon, config_test;

static list<vm_socket::ptr> to_listen;
static vm_conf::ptr conf;
static vm_dispatcher dispatcher;

vm_cmd_arg vm_all_params[] = {
  {WSTR('h'), WSTR("help"), WSTR(""), WSTR("Show this help screen.")},
  {WSTR('d'), WSTR("daemon"), WSTR(""), WSTR("Run the server in daemon mode.")},
  {WSTR('c'), WSTR("conf"), WSTR("FILE_NAME"), WSTR("Use specified configuration file.")},
  {WSTR('S'), WSTR("say"), WSTR("TEXT"), WSTR("Say TEXT and exit.")},
  {WSTR('p'), WSTR("pidfile"), WSTR("FILE_NAME"), WSTR("Put pid of server process to FILE_NAME.")},
  {WSTR('t'), WSTR("test"), WSTR(""), WSTR("Only load and show configuration information.")}
};

uint vm_all_param_count=6;

static bool load_configuration(int argc, char *argv[], int &r)
{
  if (!parse_cmd_line(argc, argv))
    {
      r=1;
      return 0;
    }

  if (is_used_cmd_param(WSTR("help")))
    {
      using namespace std;
      cout << "Voicemand is the daemon for VOICEMAN speech system." << endl;
      cout << "Version: " << VOICEMAN_VERSION << "." << endl;
      cout << "You can use following command line arguments:" << endl;
      print_cmd_args_help();
      r=0;
      return 0;
    }

  conf=new vm_conf();
  vm_string conf_file_name, log;
  if (is_used_cmd_param(WSTR("conf")))
    conf_file_name=get_cmd_param_value(WSTR("conf")); else
      conf_file_name=VOICEMAN_DEFAULT_CONFIG;
  if (!conf->load(conf_file_name, log))
    {
      using namespace std;
      cerr << "Some errors were occured while processing configuration file:" << endl;
      cerr << log;
      r=1;
      return 0;
    }

  if (is_used_cmd_param(WSTR("say")))
    config_say = get_cmd_param_value(WSTR("say"));
  if (is_used_cmd_param(WSTR("pidfile")))
    config_pidfile = get_cmd_param_value(WSTR("pidfile"));
  config_daemon=is_used_cmd_param(WSTR("daemon"));
  config_test=is_used_cmd_param(WSTR("test"));

  if (conf->check(WSTR("digits")))
    {
      vm_string d=conf->get(WSTR("digits"));
      if (d==WSTR("normal"))
	config_digits=2; else
	  if (d==WSTR("single"))
	    config_digits=1; else
	      if (d==WSTR("none"))
		config_digits=0; else
      {
	std::cerr << "\'digits\' parameter has an unknown value \'" << d << "\' in your configuration file." << std::endl;
	r=1;
	return 0;
      }
    }
  if (conf->check(WSTR("tones")))
    config_tones=conf->get(WSTR("tones")).to<bool>();
  if (conf->check(WSTR("tonesinqueue")))
    config_tones_in_queue=conf->get(WSTR("tonesinqueue")).to<bool>();
  if (conf->check(WSTR("startingmessage")))
    config_message=conf->get(WSTR("startingmessage"));
  if (conf->check(WSTR("capitalization")))
    config_cap=conf->get(WSTR("capitalization")).to<bool>();
  if (conf->check(WSTR("separation")))
    config_sep=conf->get(WSTR("separation")).to<bool>();
  if (conf->check(WSTR("maxclients")))
    config_maxclients = conf->get(WSTR("maxclients")).to<uint>();
  if (conf->check(WSTR("maxqueue")))
    config_tq_limit=conf->get(WSTR("maxqueue")).to<uint>();
  if (conf->check(WSTR("loopdelay")))
    config_tq_delay=conf->get(WSTR("loopdelay")).to<uint>();
  if (conf->check(WSTR("maxinputline")))
    config_maxline=conf->get(WSTR("maxinputline")).to<uint>();
  config_datadir=conf->get(WSTR("datadir"));
 
  if (conf->check(WSTR("logfilename")))
    config_log_file_name = conf->get(WSTR("logfilename"));
  if (conf->check(WSTR("logtostderr")))
    config_log_stderr = conf->get(WSTR("logtostderr")).to<bool>();
  if (conf->check(WSTR("loglevel")))
    config_log_info=(conf->get(WSTR("loglevel"))).to<uint>();
  if (config_daemon)
    config_log_stderr=0;
  return 1;
}

static bool init_output()
{
  int i;
  for(i=0;i<conf->m_sections.size();i++)
    {
      vm_conf::section &sec=conf->m_sections[i];
      if (sec.m_name != WSTR("output"))
	continue;

      if (!sec.check(WSTR("name")))
	{
	  vmlog << vmfatal << WSTR("There is output without name in your config file.") << vmendl;
	  return 0;
	}

      uint j;
      for(j=0;j<i;j++)
	if (conf->m_sections[j].m_name == WSTR("output") && conf->m_sections[j][WSTR("name")] == sec[WSTR("name")])
	  {
	    vmlog << vmfatal << WSTR("There are outputs with the same name \'") << sec[WSTR("name")] << WSTR("\'.") << vmendl;
				return 0;
	  } // same name

      vm_string name = sec[WSTR("name")];
      if (name == WSTR("default"))
	{
	  vmlog << vmfatal << WSTR("Name \'default\' is illegal for outputs.") << vmendl;
	  return 0;
	}
      vm_string t=sec[WSTR("type")];
      if (t==WSTR("command"))
	{
	  vm_command_player::ptr p(new vm_command_player());
	  if (!p->init(sec))
	    {
	      vmlog << vmfatal << WSTR("Could not initialize output \'") << name << WSTR("\'.") << vmendl;
	      return 0;
	    }
	  tq_add_player(p);
	  vmlog << vminfo << WSTR("Added command output \'") << name << WSTR("\'.") << vmendl;
	} else // command output
	  {
	    vmlog << vmfatal << WSTR("Output \'") << name << WSTR("\' has unknwon type \'") << t << WSTR("\'.") << vmendl;
	    return 0;
	  } // unknown type
    } // for outputs
  tq_run();
  return 1;
}

static void close_output()
{

}

static bool init_sockets()
{
  if (!conf->check(WSTR("socket")) && !conf->check(WSTR("port")))
    {
      vmlog << vmfatal << WSTR("No sockets for listening are specified.") << vmendl;
      return 0;
    }
  vm_socket::ptr ud_socket, tcpip_socket;
  if (conf->check(WSTR("socket")))
    {
      vm_unix_socket::ptr uds(new vm_unix_socket());
      if (!uds->open(vm_string2io(conf->get(WSTR("socket")))))
	{
	  vmlog << vmfatal << WSTR("Could not open UNIX domain socket as ") << conf->get(WSTR("socket")) << WSTR(".") << vmendl;
	  return 0;
	}
      ud_socket=uds;
      vmlog << vminfo << WSTR("UNIX domain socket is ready as ") << conf->get(WSTR("socket")) << WSTR(".") << vmendl;
    }

  if (conf->check(WSTR("port")))
    {
      vm_inet_socket::ptr ts(new vm_inet_socket());
      if (!ts->open(conf->get(WSTR("port")).to<uint>()))
	{
	  vmlog << vmfatal << WSTR("Could not open TCP/IP socket at port ") << conf->get(WSTR("port")).to<uint>() << WSTR(".") << vmendl;
	  return 0;
	}

      tcpip_socket=ts;
      vmlog << vminfo << WSTR("TCP/IP socket is ready at port ") << conf->get(WSTR("port")).to<uint>() << WSTR(".") << vmendl;
    }

  if (conf->check(WSTR("socket")))
    to_listen.push_back(ud_socket);
  if (conf->check(WSTR("port")))
    to_listen.push_back(tcpip_socket);
  return 1;
}

static void close_sockets()
{
  to_listen.clear();
}

static bool init_languages(vm_conf::ptr conf)
{
  for(int i=0;i<conf->m_sections.size();i++)
    {
      vm_conf::section& s=conf->m_sections[i];
      if (s.m_name != WSTR("output"))
	continue;
      if (!s.check(WSTR("name")) || s[WSTR("name")].empty())
	{
	  vmlog << vmfatal << WSTR("One of your outputs has no name.") << vmendl;
	  return 0;
	}
      vm_string name = s[WSTR("name")];
      if (!s.check(WSTR("lang")) || s[WSTR("lang")].empty())
	continue;
      language_mapper.add(name, s[WSTR("lang")]);
    }
  return 1;
}

static int run()
{
  if (!init_languages(conf))
    return 1;
  if (!init_output())
    return 1;
  if (!dispatcher.init(conf))
    return 1;
  if (!init_sockets())
    return 1;
  if (!config_pidfile.empty())
    {
      std::ofstream f(vm_string2io(config_pidfile).c_str());
      if (!f)
	vmlog << vmerror << WSTR("Could not open \'") <<config_pidfile << WSTR("\' to put pid of the server process.") << vmendl; else
	  f << getpid();
    }
  if (config_daemon)
    vmlog << vminfo << WSTR("server is ready to process speech in daemon mode.") << vmendl; else
      vmlog << vminfo << WSTR("Server is started in non-daemon mode.") << vmendl;
  if (!config_message.trim().empty())
    {
      vm_client::ptr cl=new vm_client();
      dispatcher.text(cl, config_message);
    }
  int r=dispatcher.run(to_listen);
  close_sockets();
  close_output();
  return r;
}

static int say_mode(const vm_string &text)
{
  if (text.empty())
    {
      vmlog << vmerror << WSTR("Server started in \"say\" mode, but text for processing is empty.") << vmendl;
      return 1;
    }
  if (!init_languages(conf))
    return 1;
  if (!init_output())
    return 1;
  if (!dispatcher.init(conf))
    return 1;
  vmlog << vminfo << WSTR("Server started in \"say\" mode. Text: ") << text << WSTR(".") << vmendl;
  vm_client::ptr client=new vm_client();
  tq_idle=0;
  dispatcher.text(client, config_say);
  while(!tq_idle);
  close_output();
	return 0;
}

int main(int argc, char *argv[])
{
  if (!primary_init())
    return 1;
  int r;
  if (!load_configuration(argc, argv, r))
    return r;
  if (config_test)
    {
      using namespace std;
      int i;
      for(i=0;i<conf->m_sections.size();i++)
	{
	  vm_conf::section &sec=conf->m_sections[i];
	  cout << "Section \'" << sec.m_name << "\':" << endl;
	  vm_conf::section::iterator it;
	  for(it=sec.begin();it!=sec.end();it++)
	    cout << "  " << it->first << ": " << it->second << endl;
	}
      cout << "Loaded " << conf->get_replacement_count() << " replacement(s)." << endl;
      cout << "Loaded " << conf->get_chars_table_item_count() << " characters table item(s)." << endl;
      return 0;
    }
  vmlog << vminfo << WSTR("Loading Russian constants.") << vmendl;
  vm_string log;
  if (!rus_lang.init(config_datadir+WSTR("ru_const"), log))
    {
      std::cerr << "There were errors with loading Russian constants:" << std::endl;
      std::cerr << log;
      return 1;
    }

  if (!config_say.empty())
    return say_mode(config_say);

  if (!config_daemon)
    return run();
  pid_t p=fork();
  if (p==(pid_t)-1)
    {
      using namespace std;
      cerr << "Could not create new process:" << strerror(errno) << endl;
      return 1;
    }

  if (p == (pid_t)0)
    {
      vmlog << vminfo << WSTR("New process started.") << vmendl;
      exit(run());
    }

  return 0;
}
