/*
	Copyright (c) 2000-2006 Michael Pozhidaev<msp@altlinux.org>. 
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"
#include"vmclient.h"

vm_cmd_arg vm_all_params[] = {
  {WSTR('h'), WSTR("help"), WSTR(""), WSTR("Show this help screen;")},
  {WSTR('s'), WSTR("socket"), WSTR("FILE_NAME"), WSTR("Connect to server via UNIX domain socket;")},
  {WSTR('p'), WSTR("host"), WSTR("HOST[:PORT]"), WSTR("Connect to server via TCP/IP;")},
  {WSTR('S'), WSTR("say"), WSTR("TEXT"), WSTR("Say TEXT and exit.")}
};
uint vm_all_param_count=4;

static vm_connection_t con=VOICEMAN_BAD_CONNECTION;

#define VOLUME_HEAD WSTR("volume=")
#define PITCH_HEAD WSTR("pitch=")
#define RATE_HEAD WSTR("rate=")

static bool check_begins(const vm_string &head, const vm_string &str, vm_string &tail)
{
  if (head.length() >= str.length())
    return 0;
  int i;
  for(i=0;i<head.length();i++)
    if (str[i] != head[i])
      return 0;
  tail.erase();
  for(i=head.length();i<str.length();i++)
    tail+=str[i];
  return 1;
}

static void process(const vm_string &line)
{
  if (line.empty())
    {
      vm_stop(con);
      return;
    }
  if (line.length()==1)
    {
      vm_letter(con, make_utf8(line).c_str());
      return;
    }
  vm_string s;
  if (check_begins(VOLUME_HEAD, line, s))
    {
      if (!s.is<unsigned int>())
	{
	  vmlog << vmerror << WSTR("\'") << s << WSTR("\' is not a valid unsigned integer number.") << vmendl;
	  return;
	}
      vm_volume(con, s.to<unsigned int>());
      return;
    }
  if (check_begins(PITCH_HEAD, line, s))
    {
      if (!s.is<unsigned int>())
	{
	  vmlog << vmerror << WSTR("\'") << s << WSTR("\' is not a valid unsigned integer number.") << vmendl;
	  return;
	}
      vm_pitch(con, s.to<unsigned int>());
      return;
    }
  if (check_begins(RATE_HEAD, line, s))
    {
      if (!s.is<unsigned int>())
	{
	  vmlog << vmerror << WSTR("\'") << s << WSTR("\' is not a valid unsigned integer number.") << vmendl;
	  return;
	}
      vm_rate(con, s.to<unsigned int>());
      return;
    }
  vm_text(con, make_utf8(line).c_str());
}

int main(int argc, char **argv)
{
  if (!primary_init())
    return 1;
  using std::cout;
  using std::cin;
  using std::cerr;
  using std::endl;
  if (!parse_cmd_line(argc, argv))
    return 1;
  if (is_used_cmd_param(WSTR("help")))
    {
      cout << "Shell client for VOICEMAN speech system." << endl;
      cout << endl << "Command line options:" << endl;
      print_cmd_args_help();
      return 0;
    }
  if (is_used_cmd_param(WSTR("host")))
    {
      vm_string param=get_cmd_param_value(WSTR("host"));
      vm_string h=param.substr(0);
      vm_string p=param.substr(1);
      if (h.trim().empty())
	h=WSTR("localhost");
      int port;
      if (!p.trim().empty())
	{
	  if (!p.is<unsigned int>())
	    {
	      vmlog << vmerror << WSTR("\'") << p << WSTR("\' is not a valid port number.") << vmendl;
	      return 1;
	    }
	  port = p.to<unsigned int>();
	} else 
	  port=VOICEMAN_DEFAULT_PORT;
      con=vm_connect_inet(vm_string2io(h).c_str(), port);
      if (con == VOICEMAN_BAD_CONNECTION)
	{
	  vmlog << vmerror << WSTR("Could not establish connection to \'") << h << WSTR("\' to port ") << port << WSTR(".") << vmendl;
	  return 1;
	}
    } else
      if (is_used_cmd_param(WSTR("socket")))
	{
	  vm_string p=get_cmd_param_value(WSTR("socket"));
	  if (p.trim().empty())
	    {
	      vmlog << vmerror << WSTR("Missed name of UNIX domain socket.") << vmendl;
		return 1;
	    }
	  if (p.trim() == WSTR("-"))
	    p=io2vm_string(VOICEMAN_DEFAULT_SOCKET);
	  con = vm_connect_unix(vm_string2io(p).c_str());
	  if (con == VOICEMAN_BAD_CONNECTION)
	    {
	      vmlog << vmerror << WSTR("Could not connect to server via UNIX domain socket \'") << p << WSTR("\'.") << vmendl;
	      return 1;
	    }
	} else
	  {
	    con = vm_connect();
	    if (con == VOICEMAN_BAD_CONNECTION)
	      {
		vmlog << vmerror << WSTR("Could not connect to voicemand with default settings.") << vmendl;
		  return 1;
	      }
	  }
  if (is_used_cmd_param(WSTR("say")))
    {
      vm_text(con, make_utf8(get_cmd_param_value(WSTR("say"))).c_str());
      vm_close(con);
      return 0;
    }
  cout << "VOICEMAN speech system. Ver: " << VOICEMAN_VERSION << "." << endl;
  cout << "Type \'quit\' or press Ctrl+D to leave this prompt." << endl;
  while(1)
    {
      string l;
      bool to_quit = 0;
      cout << "voiceman>";
      while(1)
	{
	  char c;
	  if (!cin.get(c))
	    {
	      to_quit = 1;
	      break;
	    }
	  if (c == 13)
	    continue;
	  if (c == 10)
	    break;
	  l+=c;
	}
      if (to_quit || (l == "quit"))
	break;
      process(io2vm_string(l));
    }
  vm_close(con);
  cout << endl << "Bye!!!" << endl;
  return 0;
}
