/*
	Copyright (c) 2000-2006 Michael Pozhidaev<msp@altlinux.org>. 
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#ifndef __VMSTRING_H__
#define __VMSTRING_H__

typedef wchar_t vm_char;
#define WSTR(x) L ## x

namespace __gnu_cxx
{
  template<> struct hash<vm_char>
  {
    size_t operator()(vm_char x) const
    {
      return hash< size_t>()(x);
    }
  };
}

class vm_string: public std::wstring
{
public:
  class iterator 
  {
    friend class vm_string;

  private:
    iterator(const vm_string &str, const vm_string &chars): m_start(-1), m_str(str), m_end(-1)
    {
	assert(!chars.empty());
	for(int i=0;i<chars.length();i++)
	  m_chars.insert(chars[i]);
    }

  public:
    bool match(vm_char c) const
    {
      hash_set<vm_char>::const_iterator i=m_chars.find(c);
      return i!=m_chars.end();
    }

    bool next()
    {
      m_start=m_end+1;
      if (m_start>=m_str.length())
	return 0;
      while(m_start<m_str.length() && !match(m_str[m_start])) m_start++;
      if (m_start>=m_str.length())
	return 0;
      m_end=m_start;
      while(m_end<m_str.length() && match(m_str[m_end])) m_end++;
      return 1;
    };

    vm_string str() const
    {
      assert(m_start>=0 && m_end <= m_str.length());
      vm_string s;
      for(int i=m_start;i<m_end;i++)
	s+=m_str[i];
      return s;
    }

    int start() const { assert(m_start>=0 && m_end<=m_str.length()); return m_start; }
    int end() const { assert(m_start>=0 && m_end <=m_str.length()); return m_end; }

  private:
    const vm_string &m_str;
    hash_set<vm_char> m_chars;
    int m_start, m_end;
  };//class iterator;

  vm_string() {}
  vm_string(const wchar_t *s) : std::wstring(s) {}
  vm_string(const std::wstring &s) : std::wstring(s) {}
  vm_string(double f, unsigned long p);
  const vm_string &operator =(const vm_char s);
  iterator create_iterator(const vm_string &chars) const {return iterator(*this, chars);}
  bool contains(const vm_string &s) const;
  vm_string trim() const;
  vm_string upper_case() const;
  vm_string lower_case() const;
  template<typename T> bool is()const;
  template<typename T> T to() const;
  bool is_double() const;
  double to_double() const;
  void space_undup_attach(vm_char c);
  void space_undup_attach(const vm_string &s);
  void space_undup();
  vm_string substr(int index) const;
}; // class vm-string;

typedef std::wostringstream vm_ostringstream;
typedef std::wistringstream vm_istringstream;

#define io2vm_string io2wstring
#define vm_string2io wstring2io
#define make_utf8 wstring2utf8
#define decode_utf8 utf82wstring
#define decode1_utf8 utf82wstring1

std::ostream &operator <<(std::ostream &s, const wstring &ws);

wstring io2wstring(const string &s);
string wstring2io(const wstring &s);
string wstring2utf8(const wstring &s);
wstring utf82wstring(const string &s);
bool utf82wstring1(const string &s, wstring &res);
bool primary_init();

#define ENG_LETTERS WSTR("abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ")

#endif //__VMSTRING_H__
