/*
	Copyright (c) 2000-2006 Michael Pozhidaev<msp@altlinux.org>. 
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"

#define VOICEMAN_ERROR_EXIT_CODE 2

vm_log vmlog;
vm_info vminfo;
vm_warn vmwarn;
vm_error vmerror;
vm_fatal vmfatal;
vm_endl vmendl;

vm_string config_log_file_name;
bool config_log_stderr=1, config_log_info=0;

static vm_string get_current_time()
{
  vm_string s;
  time_t t;
  time(&t);
  s=io2vm_string(ctime(&t));
  uint i;
  vm_string ss;
  for(i=0;i<s.length();i++)
    if (s[i]!=10 && s[i]!=13)
      ss+=s[i];
  return ss;
}

static void dispatch_log_string(const vm_string &s)
{
  if (config_log_stderr)
    std::cerr << s << std::endl;
  if (!config_log_file_name.empty())
    {
      std::ofstream f(vm_string2io(config_log_file_name).c_str(), std::ios_base::out | std::ios_base::app);
      if (f)
	f << get_current_time() << ':' << s << std::endl;
    }
}

void vm_log::begin()
{
  if (!m_stream)
    delete m_stream;

  m_stream = new vm_ostringstream();
}

void vm_log::close()
{
  if (m_stream)
    {
      delete m_stream;
      m_stream = NULL;
    }
}

vm_log &vm_log::operator <<(int n)
{
  if (!m_stream)
    return *this;

  vm_ostringstream &s=*m_stream;
  s<<n;
  return *this;
}

vm_log &vm_log::operator <<(uint n)
{
  if (!m_stream)
    return *this;

  vm_ostringstream &s=*m_stream;
  s<<n;
  return *this;
}

vm_log &vm_log::operator <<(const char *s)
{
  if (!m_stream)
    return *this;

  vm_ostringstream &ss=*m_stream;
  ss<<io2vm_string(s);
  return *this;
}

vm_log &vm_log::operator <<(const vm_string &s)
{
  if (!m_stream)
    return *this;

  vm_ostringstream &ss=*m_stream;
  ss<<s;
  return *this;
}

vm_log &vm_log::operator <<(const vm_info &c)
{
  begin();
  if (!config_log_info)
    {
      close();
		return *this;
    }

  vm_ostringstream &s=*m_stream;
  s<<WSTR("INF:"); 
  return *this;
}

vm_log &vm_log::operator <<(const vm_warn &c)
{
  begin();
  vm_ostringstream &s=*m_stream;
  s << WSTR("WARNING:");
  return *this;
}

vm_log &vm_log::operator <<(const vm_error &c)
{
  begin();
  vm_ostringstream &s=*m_stream;
  s << WSTR("ERROR:");
  return *this;
}

vm_log &vm_log::operator <<(const vm_fatal &c)
{
  begin();
  vm_ostringstream &s=*m_stream;
  s << WSTR("FATAL:");
  return *this;
}

vm_log &vm_log::operator <<(const vm_endl &c)
{
  if (!m_stream)
    return *this;
  dispatch_log_string(m_stream->str());
  close();
  return *this;
}

#ifndef VOICEMAN_DEBUG
void vm_stop(const vm_char *msg)
#else
void vm_stop(const vm_char *msg, const char *file, int line)
#endif // VOICEMAN_DEBUG
{
  vmlog << vmfatal << msg ;
#ifdef VOICEMAN_DEBUG
  vmlog << " (" << file << "(" << line << "))";
#endif // VOICEMAN_DEBUG
  vmlog << vmendl;
  vmlog.close();
  exit(VOICEMAN_ERROR_EXIT_CODE);
}

#ifndef VOICEMAN_DEBUG
void vm_sys_stop(const vm_char *msg)
#else
void vm_sys_stop(const vm_char *msg, const char *file, int line)
#endif // VOICEMAN_DEBUG
{
  vmlog << vmfatal << msg << WSTR(":") << strerror(errno) << WSTR(".");
#ifdef VOICEMAN_DEBUG
  vmlog << WSTR(" (") << file << WSTR("(") << line << WSTR("))");
#endif // VOICEMAN_DEBUG
  vmlog << vmendl;
  vmlog.close();
  exit(VOICEMAN_ERROR_EXIT_CODE);
}

#ifndef VOICEMAN_DEBUG
void vm_sys_warn(const vm_char *msg)
#else
void vm_sys_warn(const vm_char *msg, const char *file, int line)
#endif // VOICEMAN_DEBUG
{
  vmlog << vmwarn << msg << WSTR(":") << strerror(errno) << WSTR(".");
#ifdef VOICEMAN_DEBUG
  vmlog << WSTR(" (") << file << WSTR("(") << line << WSTR("))");
#endif
}
