/*
	Copyright (c) 2000-2006 Michael Pozhidaev<msp@altlinux.org>. 
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"

void vm_socket::close()
{
  if (!m_opened)
    return;
  ::close(m_sock);
  m_opened=0;
};

int vm_socket::get_fd() const
{
  if (!m_opened)
    VM_STOP("Could not return handle to closed socket.");
  return m_sock;
}

int vm_socket::read(void *buf, int s)
{
  if (!m_opened)
    VM_STOP("Could not read data from closed socket.");
  int r = ::read(m_sock, buf, s);
  if (r==-1)
    VM_SYS_STOP("read()");
  return r;
}

int vm_socket::write(const void *buf, int s)
{
  if (!m_opened)
    VM_STOP("Could not write data to closed socket.");
  int r = ::write(m_sock, buf, s);
  if (r==-1)
    VM_SYS_STOP("write()");
  return r;
}

bool vm_socket::read(string &s)
{
  char buf[2048];
  int c=read(buf, sizeof(buf));
  if (c==0)
    return 0;
  uint i;
  s.erase();
  for(i=0;i<c;i++)
    s+=buf[i];
  return 1;
}

void vm_socket::write(const string &s)
{
  uint k = s.length();
  do {
    char buf[2048];
    int sent_count;
    uint i;
    if (k >= sizeof(buf))
      {
	for(i=0;i<sizeof(buf);i++)
	  buf[i] = s[s.length()-k+i];
	sent_count = write(buf, sizeof(buf));
	k-=sent_count;
      } else
	{
	  for(i=0;i<k;i++)
	    buf[i]=s[s.length()-k+i];
	  sent_count = write(buf, k);
	  k -= sent_count;
	}
  } while(k);
}

bool vm_unix_socket::open(const string &name)
{
  if (m_opened)
    VM_STOP("Socket is already opened.");

  unlink(name.c_str());
  struct sockaddr_un addr;
  int fd;
  fd = socket(AF_UNIX, SOCK_STREAM, 0);
  if (fd == -1)
    {
      VM_SYS_WARN("socket(AF_UNIX, SOCK_STREAM, 0)");
      return 0;
    }
  addr.sun_family=AF_UNIX;
  strncpy(addr.sun_path, name.c_str(), sizeof(addr.sun_path));
  if (bind(fd, (struct sockaddr*)&addr, sizeof(sockaddr_un)) == -1)
    {
      ::close(fd);
      VM_SYS_WARN("bind()");
      return 0;
    }

  if (listen(fd, 512) == -1)
    {
      ::close(fd);
      VM_SYS_WARN("listen()");
      return 0;
    }
  chmod(name.c_str(), 0666);
  m_sock=fd;
  m_opened=1;
  return 1;
}

bool vm_inet_socket::open(uint port)
{
  if (m_opened)
    VM_STOP("Socket is already opened.");

  int fd;
  struct sockaddr_in addr;
  fd=socket(AF_INET, SOCK_STREAM, 0);
  if (fd == -1)
    {
      VM_SYS_WARN("socket(AF_INET, SOCK_STREAM, 0)");
      return 0;
    }
  bzero(&addr, sizeof(struct sockaddr_in));
  addr.sin_family=AF_INET;
  addr.sin_addr.s_addr=htonl(INADDR_ANY);
  addr.sin_port=htons(5511);
  if (bind(fd, (struct sockaddr*)&addr, sizeof(struct sockaddr_in)) == -1)
    {
      ::close(fd);
      VM_SYS_WARN("bind()");
      return 0;
    }
  if (listen(fd, 512) == -1)
    {
      ::close(fd);
      VM_SYS_WARN("listen()");
      return 0;
    }
  m_sock=fd;
  m_opened=1;
  return 1;
}
