/*
	Copyright (c) 2000-2006 Michael Pozhidaev<msp@altlinux.org>. 
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#ifndef __LANGS_H__
#define __LANGS_H__

#include"confparse.h"

#define IS_DIGIT(x) ((x >= WSTR('0')) && (x <= WSTR(x)))

class vm_lang
{
public:
  virtual vm_string get_chars() const = 0;
  virtual uint get_char_type(vm_char c) const = 0;
  virtual vm_string insert_digits(const vm_string &in_str, bool single_digits) const = 0;
  virtual vm_string up_case(const vm_string &str) const = 0;
  virtual vm_string low_case(const vm_string &str) const = 0;
  virtual bool equal_chars(vm_char c1, vm_char c2) const;
  virtual bool equal_strings(const vm_string &s1, const vm_string &s2) const;
  virtual vm_string space_separation(const vm_string& text) const {return text;};
  virtual void mark_caps(const vm_string &text, std::vector<bool> &mask) {};
};

class vm_eng_lang : public vm_lang
{
public:
  virtual vm_string get_chars() const;
  virtual uint get_char_type(vm_char c) const;
  virtual vm_string insert_digits(const vm_string &in_str, bool single_digits) const;
  virtual vm_string up_case(const vm_string &str) const;
  virtual vm_string low_case(const vm_string &str) const;
  virtual vm_string space_separation(const vm_string& text) const;
  virtual void mark_caps(const vm_string &text, std::vector<bool> &mask);
  void add_cap_item(const vm_string& str, bool before, bool after) {m_cap_items.push_back(vm_cap_item(str, before, after));}

private:
  class vm_cap_item
  {
  public:
    vm_cap_item(): m_before(0), m_after(0) {};
    vm_cap_item(const vm_string& str, bool before, bool after): m_str(str), m_before(before), m_after(after) {};
    vm_string m_str;
    bool m_before, m_after;
  }; // class vm-cap_item;

private:  
  vm_string process_hundred(const vm_string &in_str, vm_char *items[]) const;
  vm_string digits_to_words(const vm_string &in_str) const;
  bool check_cap_list(const vm_string &str, uint pos, vm_string& result) const;
  void process_cap_list(const vm_string& str, std::vector<bool>& mask) const;

private:  
  std::list<vm_cap_item> m_cap_items;
};

class vm_rus_lang : public vm_lang
{
public:
  virtual vm_string get_chars() const;
  virtual uint get_char_type(vm_char c) const;
  virtual vm_string insert_digits(const vm_string &in_str, bool single_digits) const;
  virtual vm_string up_case(const vm_string &str) const;
  virtual vm_string low_case(const vm_string &str) const;
  bool init(const vm_string &file_name, vm_string &log);

private:
  class rus_char 
  {
  public:
    rus_char() {};
    rus_char(vm_char up_c, vm_char low_c): up_char(up_c), low_char(low_c) {};
    vm_char up_char, low_char;
  };

private:  
  bool init_vector(const vm_conf_parser::section &sec, std::vector<vm_string> &v, vm_string &log);
  bool init_map(const vm_conf_parser::section &sec, std::vector<rus_char> &v, vm_string &log);
  vm_string process_hundred(const vm_string &in_str, const std::vector<vm_string> &items, bool female) const;
  vm_string digits_to_words(const vm_string &in_str) const;

private:
  vm_string m_zero, m_chars;
  std::vector<vm_string> rus_mlrds, rus_mlns, rus_thnds, rus_hundreds, rus_decimals, rus_tens, rus_ones, rus_ones_f;
  std::vector<rus_char> rus_chars;
};

extern vm_eng_lang eng_lang;
extern vm_rus_lang rus_lang;

class vm_language_mapper
{
public:
  void add(const vm_string &output, const vm_string &lang);
  vm_lang *operator [](const vm_string &output_name) const;
private:
  std::map<vm_string, vm_string> m_info;
};

extern vm_language_mapper language_mapper;

#endif // __LANGS_H__
