/*
	Copyright (c) 2000-2006 Michael Pozhidaev<msp@altlinux.org>. 
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"
#include"langs.h"
#include"tq.h"
#include"proc.h"

int config_digits=2;
bool config_cap=1, config_sep=1;

std::map<vm_char, vm_string> vm_chars_table;

void vm_language_mapper::add(const vm_string &output, const vm_string &lang)
{
  if (output.empty() || lang.empty())
    return;
  m_info[output]=lang;
}

vm_lang *vm_language_mapper::operator [](const vm_string &output_name) const
{
  std::map<vm_string, vm_string>::const_iterator i=m_info.find(output_name);
  if (i==m_info.end())
    {
      vmlog << vmerror << WSTR("Output \'") << output_name << WSTR("\' has no language information.") << vmendl;
      return NULL;
    }
  const vm_string &lang_name=i->second;
  if (lang_name==WSTR("eng"))
    return &eng_lang; else
      if (lang_name==WSTR("rus"))
	return &rus_lang; else
	  vmlog << vmerror << WSTR("Output \'") << output_name << WSTR("\' uses unsupported language \'") << lang_name << WSTR("\'.") << vmendl;
  return NULL;
}

void vm_text_splitter::set_map_items(const vm_string &output_name, const vm_string &items)
{
  int i;
  for(i=0;i<items.length();i++)
    m_map[items[i]]=output_name;
}

void vm_text_splitter::split_text(const vm_string &text, vector<text_item> &items) /*const*/
{
  int i;
  text_item c_item;
  vm_string c_default;
  vm_string output_name;
  items.clear();
  for(i=0;i<text.length();i++)
    {
      vm_char let = text[i];
      if (let == WSTR(' '))
	{
	  if (c_item.m_output.empty())
	    continue;
	  if (c_item.m_text.empty())
	    continue;
	  if (c_item.m_text[c_item.m_text.length()-1] == WSTR(' '))
	    continue;
	  c_item.m_text += WSTR(' ');
	  continue;
	} // space
      output_name = m_map[let];
      if (output_name.empty())
	{
	  if (c_item.m_text.empty())
	    continue;
	  if (c_item.m_text[c_item.m_text.length()-1]== WSTR(' '))
	    continue;
	  c_item.m_text += WSTR(' ');
	  continue;
	}
      if (output_name == WSTR("default"))
	{
	  if (c_default.empty())
	    output_name = m_default_output; else
	      output_name = c_default;
	}
      if ((!c_item.m_output.empty()) && (c_item.m_output != output_name))
	{
	  items.push_back(c_item);
	  c_item = text_item();
	}
      c_default = output_name;
      c_item.m_output = output_name;
      c_item.m_text += let;
    }
  if (!c_item.m_output.empty())
    items.push_back(c_item);
}

void vm_text_replacer::add_replacement(const vm_string &output, const vm_string &old_str, const vm_string &new_str)
{
  if (output.empty() || old_str.empty() || new_str.empty())
    return;
  m_replacements.push_back(replacement(output, old_str, new_str));
}

bool vm_text_replacer::get_replacement(const vm_string &str, uint pos, const vm_string &output, uint &result) const
{
  uint i;
  vm_lang *lang=language_mapper[output];
  if (!lang)
    return 0;
  for(i=0;i<m_replacements.size();i++)
    {
      uint j;
      bool b = 1;
      if (m_replacements[i].m_output != output)
	continue;
      if (str.length() - pos < m_replacements[i].m_old_str.length())
	continue;
      for(j=0;j<m_replacements[i].m_old_str.length();j++)
	if (!lang->equal_chars(str[pos+j], m_replacements[i].m_old_str[j]))
	  {
	    b = 0;
	    break;
	  }
      if (b)
	{
	  result = i;
	  return 1;
	}
    }
  return 0;
}

vm_string vm_text_replacer::insert_replacements(const vm_string &str, const vm_string &output) const
{
  uint i, k;
  vm_string new_str;
  for(i=0;i<str.length();i++)
    {
      if (get_replacement(str, i, output, k))
	{
	  if (!new_str.empty() && new_str[new_str.length()-1]!= WSTR(' ')) new_str+=WSTR(' ');
	  new_str += m_replacements[k].m_new_str;
	  if (!new_str.empty() && new_str[new_str.length()-1]!= WSTR(' ')) new_str+=WSTR(' ');
	  i += m_replacements[k].m_old_str.length()-1;
	} else
	  {
	    if (str[i]==WSTR(' '))
	      {
		if (!new_str.empty() && new_str[new_str.length()-1]!= WSTR(' '))
		  new_str += WSTR(' ');
	      } else
		new_str += str[i];
	  }
    }
  return new_str;
}

void vm_processor::letter(vm_char c, vm_param volume, vm_param pitch, vm_param rate)
{
  if (!m_splitter)
    {
      vmlog << vmerror << WSTR("Text splitter is not created.") << vmendl;
      return;
    }
  vm_string name=m_splitter->m_map[c];
  if (name==WSTR("default"))
    name=m_splitter->m_default_output;

  if (name.empty())
    {
      if (!vm_chars_table[c].empty())
	line(vm_chars_table[c], volume, pitch, rate);
      return;
    }
  vm_string s;
  std::vector<bool> mask;
  if (!vm_chars_table[c].empty())
    {
      s=vm_chars_table[c];
      mask.resize(s.length());
      for(int i=0;i<mask.size();i++)
	mask[i]=0;
    } else
      {
	s+=c;
	mask.push_back(1);
      }
  vm_param p=pitch;
  vm_lang *l=language_mapper[name];
  if (l && l->get_char_type(c)==2)
    p.increase(0.3);
  text_out(name, s, mask, volume, p, rate);
}

void vm_processor::line(const vm_string &text, vm_param volume, vm_param pitch, vm_param rate)
{
  vector<vm_text_splitter::text_item> items;
  if (!m_splitter)
    {
      vmlog << vmerror << WSTR("Processor has no text splitter. Igmoring text line.") << vmendl;
      return;
    }
  if (!m_replacer)
    {
      vmlog << vmerror << WSTR("Processor has no text replacer.") << vmendl;
      return;
    }
  m_splitter->split_text(text, items);
  int i;
  for(i=0;i<items.size();i++)
    {
      vector<bool> mask;
      vm_string to_send=m_replacer->insert_replacements(items[i].m_text, items[i].m_output);
      vm_lang *lang=language_mapper[items[i].m_output];
      if (!lang)
	continue;
      if (config_sep)
	to_send=lang->space_separation(to_send);
      if (config_digits==2)
	to_send=lang->insert_digits(to_send, 0); else
	  if (config_digits==1)
	    to_send=lang->insert_digits(to_send, 1);
      mask.resize(to_send.length());
      for(std::vector<bool>::iterator it=mask.begin();it!=mask.end();it++)
	*it=0;
      if (config_cap)
      lang->mark_caps(to_send, mask);
      text_out(items[i].m_output, to_send, mask, volume, pitch, rate);
    }
}

void vm_processor::text_out(const vm_string &output, const vm_string &text, const vector<bool> &mask, vm_param volume, vm_param pitch, vm_param rate) const
{
  tq_add_text(output, text, volume, pitch, rate, mask);
}
