/*
	Copyright (c) 2000-2006 Michael Pozhidaev<msp@altlinux.org>. 
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"
#include"tq.h"
#include"player.h"
#include"tone.h"

int config_tq_limit=500, config_tq_delay=50000;
volatile bool tq_idle=1;

class vm_tq_item
{
public:
  enum {TEXT=0, TONE=1};

  vm_tq_item(): m_type(TEXT), m_freq(0), m_lengthms(0) {}
  vm_tq_item(int type): m_type(type), m_freq(0), m_lengthms(0) {}
  virtual ~vm_tq_item() {}

  int m_type;
  vm_string m_name, m_text;
  vm_param m_volume, m_pitch, m_rate;
  std::vector<bool> cap_mask;
  int m_freq, m_lengthms;
};

static list<vm_tq_item> tq;
static pthread_mutex_t tq_mutex;
static pthread_t tq_thread;
static vector<vm_player::ptr> tq_players;
static vm_tone tone;

static void *tq_process(void *param)
{
  while(1)
    {
      usleep(config_tq_delay);
      pthread_mutex_lock(&tq_mutex);
      if (tq.size()==0)
	{
	  if (!tq_idle)
	    {
	      vector<vm_player::ptr>::iterator i;
	      bool b = 0;
	      for(i=tq_players.begin();i!=tq_players.end();i++)
		if ((*i)->processing())
		  b = 1;
	      if (!b)
		tq_idle=1;
	    } // idle
	  pthread_mutex_unlock(&tq_mutex);
	  continue;
	}
      vector<vm_player::ptr>::iterator i;
      bool b = 0;
      for(i=tq_players.begin();i!=tq_players.end();i++)
	if ((*i)->processing())
	  b = 1;
      if (b)
	{
	  pthread_mutex_unlock(&tq_mutex);
	  continue;
	}
      vm_tq_item item=tq.front();
      tq.pop_front();
      switch(item.m_type)
	{
	case vm_tq_item::TEXT:
	  for(i=tq_players.begin();i!=tq_players.end();i++)
	    {
	      if ((*i)->m_name == item.m_name)
		{
		  (*i)->play(item.m_text, item.m_volume, item.m_pitch, item.m_rate, item.cap_mask);
		  break;
		} // if name
	    } // for players
	  break;

	case vm_tq_item::TONE:
	  tone.play(item.m_freq, item.m_lengthms);
	  break;

	default:
	  vmlog << vmwarn << WSTR("There was item with unknown type code \'") << item.m_type << WSTR("\' in text queue.") << vmendl;
	} // switch;
      pthread_mutex_unlock(&tq_mutex);
    } // endless loop
}

void tq_add_player(vm_player::ptr player)
{
  if (!player)
    VM_STOP("Could not add NULL pointer.");
  if (player->m_name.empty())
    VM_STOP("Could not add player without name.");
  tq_players.push_back(player);
}

void tq_run()
{
  VM_SYS(pthread_mutex_init(&tq_mutex, NULL) != -1, "pthread_mutex_init()");
  VM_SYS(pthread_create(&tq_thread, NULL, tq_process, NULL) != -1, "pthread_create()");
}

void tq_add_text(const vm_string &name, const vm_string &text, vm_param volume, vm_param pitch, vm_param rate, const std::vector<bool> &cap_mask)
{
  pthread_mutex_lock(&tq_mutex);
  if (config_tq_limit > 0 && tq.size() >= config_tq_limit)
    {
      vmlog << vmwarn << WSTR("Text \'") << text << WSTR("\' was sent to output \'") << name << WSTR("\', but text queue is overflowed. This text will be ignored.") << vmendl;
      pthread_mutex_unlock(&tq_mutex);
      return;
    }
#ifdef VOICEMAN_DEBUG
  vmlog << vminfo << WSTR("Sending text \'") << text << WSTR("\' to output \'") << name << WSTR("\'...") << vmendl;
#endif
  vm_tq_item p;
  p.m_name = name;
  p.m_text = text;
  p.m_volume=volume;
  p.m_pitch=pitch;
  p.m_rate=rate;
  p.cap_mask=cap_mask;
  tq.push_back(p);
  pthread_mutex_unlock(&tq_mutex);
}

void tq_add_tone(int freq, int lengthms)
{
  assert(freq >= 10);
  assert(lengthms >= 0);
  pthread_mutex_lock(&tq_mutex);
  if (config_tq_limit > 0 && tq.size() >= config_tq_limit)
    {
      vmlog << vmwarn << WSTR("Text queue limit was reached. Ignoring tone.") << vmendl;
      pthread_mutex_unlock(&tq_mutex);
      return;
    }
#ifdef VOICEMAN_DEBUG
  vmlog << vminfo << WSTR("Sending tone.") << vmendl;
#endif
  vm_tq_item p(vm_tq_item::TONE);
  p.m_freq=freq;
  p.m_lengthms=lengthms;
  tq.push_back(p);
  pthread_mutex_unlock(&tq_mutex);
}

void tq_stop()
{
  pthread_mutex_lock(&tq_mutex);
  tq.clear();
  vector<vm_player::ptr>::iterator i;
  for(i=tq_players.begin();i!= tq_players.end();i++)
    (*i)->stop();
  pthread_mutex_unlock(&tq_mutex);
}
