/*
	Copyright (c) 2000-2006 Michael Pozhidaev<msp@altlinux.org>. 
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"
#include"vmclient.h"

#define IGNORABLE_COMMANDS WSTR("tts_sync_state tts_pause tts_set_punctuations tts_set_speech_rate")

vm_cmd_arg vm_all_params[] = {
  {WSTR('h'), WSTR("help"), WSTR(""), WSTR("Show this help screen;")},
  {WSTR('t'), WSTR("test"), WSTR(""), WSTR("Perform client test.")}
};
uint vm_all_param_count=2;

static vm_connection_t con = VOICEMAN_BAD_CONNECTION;

static void text(const vm_string& t)
{
  std::string s=make_utf8(t);
  if (con==VOICEMAN_BAD_CONNECTION)
    con=vm_connect();
  if (con != VOICEMAN_BAD_CONNECTION)
    vm_text(con, s.c_str());
}

static void letter(vm_char l)
{
  vm_string q;
  q+=l;
  std::string s=make_utf8(q);
  if (con == VOICEMAN_BAD_CONNECTION)
    con=vm_connect();
  if (con != VOICEMAN_BAD_CONNECTION)
    vm_letter(con, s.c_str());
}

static void stop()
{
  if (con == VOICEMAN_BAD_CONNECTION)
    con = vm_connect();
  if (con != VOICEMAN_BAD_CONNECTION)
    vm_stop(con);
}

static void tone(int freq, int lengthms)
{
  if (con == VOICEMAN_BAD_CONNECTION)
    con=vm_connect();
  if (con != VOICEMAN_BAD_CONNECTION)
    vm_tone(con, freq, lengthms);
}

class vm_espeak_line
{
public:
  vm_espeak_line() : m_br_level(0) {}

  vm_string m_cmd;
  std::vector<vm_string> m_params;
  int m_br_level;
};

static void remove_dectalk_codes(vm_string& str)
{
  int k = 0;
  vm_string s;
  while(k < str.length())
    {
      if (str[k] == WSTR('['))
	{
	  k++;
	  while((k < str.length()) && (str[k] != WSTR(']'))) k++;
	  if ( k >= str.length())
	    {
	      str = s;
	      return;
	    }
	  k++;
	  continue;
	}
      s += str[k++];
    }
  str = s;
}

static void split_espeak_line(const vm_string& str, vm_espeak_line& line)
{
  int &level = line.m_br_level;
  vm_string next;
  bool to_params;
  if (level)
    {
      if (line.m_params.size())
	{
	  next = line.m_params[line.m_params.size()-1];
	  line.m_params.pop_back();
	  to_params = 1;
	} else
	  {
	    next = line.m_cmd;
	    line.m_params.clear();
	    line.m_cmd.erase();
	    to_params = 0;
	  }
    } else
      {
	line.m_cmd.erase();
	line.m_params.clear();
	to_params = 0;
      }
  int i;
  int state;
  if (level)
    state = 4; else
      state = 2;
  for(i=0;i<str.length();i++)
    {
      if (str[i] == WSTR(' '))
	{
	  if (state == 2)
	    continue;
	  if (state == 1)
	    {
	      state = 2;
	      continue;
	    }
	  if (state == 4)
	    {
	      next+=WSTR(' ');
	      continue;
	    }
	  if ((state == 0) || (state == 5))
	    {
	      if (to_params)
		line.m_params.push_back(next); else
		  line.m_cmd = next;
	      next.erase();
	      to_params=1;
	      state = 1;
	      continue;
	    }
	  VM_STOP("Bad state value.");
	} // spaces
      if (str[i]==WSTR('{'))
	{
	  if ((state==1) || (state == 2))
	    {
	      next.erase();
	      state = 4;
	      level=1;
	      continue;
	    }
	  if (state == 4)
	    {
	      next+=str[i];
	      level++;
	      continue;
	    }
	  if (state == 0)
	    {
	      next+=str[i]; 
	      continue;
	    }
	  continue;
	} // opening bracket
      if (str[i] == WSTR('}'))
	{
	  if (state == 4)
	    {
	      if (level==1)
		{
		  state=5;
		  level=0;
		} else
		  if (level > 1)
		    {
		      next+=str[i]; 
		      level--;
		    } else
		      {
			VM_STOP("Bad level in string.");
		      }
	      continue;
	    }
	  if (state == 0)
	    {
	      next+=str[i];
	      continue;
	    }
	  continue;
	} // closing bracket
      // any char
      if ((state==0) || (state==4))
	{
	  next+=str[i]; 
	  continue;
	}
      if ((state == 1) || (state == 2))
	{
	  next = str[i]; 
	  state = 0;
	}
    } // for
  if ((state == 0) || (state == 4) || (state== 5))
    {
      if (to_params)
	line.m_params.push_back(next); else
	  line.m_cmd = next;
    }
}

static void handle_espeak_line(const vm_espeak_line& line, vm_string& queue)
{
  if (line.m_cmd == WSTR("l"))
    {
      if (line.m_params.size()==0)
	{
	  vmlog << vmwarn << WSTR("Received LETTER command without argument.") << vmendl;
	  return;
	}
      if (line.m_params[0].empty())
	{
	  vmlog << vmwarn << WSTR("Received LETTER command, but first argument is empty.") << vmendl;
	  return;
	}
      letter(line.m_params[0][0]);
	return;
    }
  if (line.m_cmd == WSTR("q"))
    {
      if (line.m_params.size() == 0)
	{
	  vmlog << vmwarn << WSTR("Received QUEUE command, but without arguments.") << vmendl;
	  return;
	}
      queue += line.m_params[0];
      return;
    }
  if (line.m_cmd == WSTR("s"))
    {
      stop();
      queue.erase();
      return;
    }
  if (line.m_cmd == WSTR("d"))
    {
      remove_dectalk_codes(queue);
      text(queue);
      queue.erase();
      return;
    }
  if (line.m_cmd == WSTR("tts_say"))
    {
      if (line.m_params.size() == 0)
	{
	  vmlog << vmwarn << WSTR("Received \'tts_say\' command without arguments.") << vmendl;
	  return;
	}
      vm_string tt=line.m_params[0];
      remove_dectalk_codes(tt);
      text(tt);
      return;
    }
  if (line.m_cmd == WSTR("t"))
    {
      if (line.m_params.size() < 2)
	{
	  vmlog << vmerror << WSTR("Command \'t\' requires 2 parameters, but received ") << line.m_params.size() << vmendl;
	  return;
	}
      if (!line.m_params[0].is<unsigned int>())
	{
	  vmlog << vmerror << WSTR("Command \'t\' has bad first parameter: \'") << line.m_params[0] << WSTR("\'.") << vmendl;
	  return;
	}
      if (!line.m_params[1].is<unsigned int>())
	{
	  vmlog << vmerror << WSTR("Command \'t\' has bad second parameter: \'") << line.m_params[1] << WSTR("\'.") << vmendl;
	  return;
	}
      tone(line.m_params[0].to<unsigned int>(), line.m_params[1].to<unsigned int>());
      return;
    }
  vm_string ignoring(IGNORABLE_COMMANDS);
  
vm_string::iterator it=ignoring.create_iterator(vm_string(ENG_LETTERS)+WSTR("_"));
  while(it.next())
    if (it.str() == line.m_cmd)
      return;
  vmlog << vmwarn << WSTR("Received unknown command: ") << line.m_cmd << WSTR(".") << vmendl;
}

static void handle_line(const string &line)
{
  if (line.empty())
    return;
  static vm_espeak_line el;
  static vm_string queue;
  split_espeak_line(io2vm_string(line), el);
  if (el.m_br_level)
    return;
  handle_espeak_line(el, queue);
}

static void read_input(int file)
{
  string line;
  while(1)
    {
      char buf[2048];
      int read_count;
      read_count = read(file, buf, sizeof(buf));
      if (read_count < 0)
	VM_SYS_STOP("read()");
      if (read_count==0)
	return;
      int i;
      for(i=0;i<read_count;i++)
	{
	  if (buf[i]==10)
	    {
	      handle_line(line);
	      line.erase();
	      continue;
	    }
	  if (buf[i] == 13)
	    continue;
	  line+=buf[i];
	}
    }
}

int main(int argc, char *argv[])
{
  if (!primary_init())
    return 1;
  for(int k=1;k<argc;k++)
    {
      std::string param=argv[k];
      if (param=="--help" || param=="-h")
	{
	  using namespace std;
	  cout << "VoiceMan client for Emacspeak." << endl;
	  cout << "Parameters:" << endl;
	  print_cmd_args_help();
	  return 0;
	}
      if (param=="--test" || param=="-t")
	{
	  std::cout << "Testing..." << std::endl;
	  tone(220, 150);
	  tone(220, 150);
	  tone(220, 150);
	  text(WSTR("Performing client test..."));
	  tone(220, 150);
	  tone(220, 150);
	  tone(220, 150);
	  return 0;
	}
    }
  char *hd=getenv("HOME");
  if (hd==NULL)
    VM_STOP("Environment variable HOME is not set, but it is required for this program.");
  vm_string home_dir = io2vm_string(hd);
  config_log_file_name=home_dir+WSTR("/voiceman-emacspeak.log");
  config_log_stderr=0;
#ifdef VOICEMAN_DEBUG
  config_log_info = 1;
#else
  config_log_info = 0;
#endif // VOICEMAN_DEBUG
  read_input(0);
  vm_close(con);
  return 0;
}
