/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include<iostream>
#include<string>
#include<sstream>
#include<ext/hash_set>
#include<vector>
#include<assert.h>
#include<iconv.h>
#include"vmclient.h"
#include"transcoding.h"
#include"strings.h"
#include"buildConfig.h"

#define IGNORABLE_COMMANDS "tts_sync_state tts_pause tts_set_punctuations tts_set_speech_rate"

static vm_connection_t con = VOICEMAN_BAD_CONNECTION;

static void text(const std::string& t)
{
  std::string s=encodeUTF8(IO2WString(t));
  if (con==VOICEMAN_BAD_CONNECTION)
    con=vm_connect();
  if (con != VOICEMAN_BAD_CONNECTION)
    vm_text(con, s.c_str());
}

static void letter(const std::string& l)
{
  std::wstring t=IO2WString(l);
  if (t.length()>1)
    t.resize(1);
  std::string s=encodeUTF8(t);
  if (con == VOICEMAN_BAD_CONNECTION)
    con=vm_connect();
  if (con != VOICEMAN_BAD_CONNECTION)
    vm_letter(con, s.c_str());
}

static void stop()
{
  if (con == VOICEMAN_BAD_CONNECTION)
    con = vm_connect();
  if (con != VOICEMAN_BAD_CONNECTION)
    vm_stop(con);
}

static void tone(int freq, int lengthMs)
{
  if (con == VOICEMAN_BAD_CONNECTION)
    con=vm_connect();
  if (con != VOICEMAN_BAD_CONNECTION)
    vm_tone(con, freq, lengthMs);
}

struct ESPEAKLINE
{
  ESPEAKLINE(): brLevel(0) {}

  std::string cmd;
  std::vector<std::string> params;
  int brLevel;
}; //struct ESPEAKLINE;


static void removeDectalkCodes(std::string& str)
{
  int k = 0;
  std::string s;
  while(k < str.length())
    {
      if (str[k] == '[')
	{
	  k++;
	  while((k < str.length()) && (str[k] != ']')) k++;
	  if ( k >= str.length())
	    {
	      str = s;
	      return;
	    }
	  k++;
	  continue;
	}
      s += str[k++];
    }
  str = s;
}

static void splitEspeakLine(const std::string& str, ESPEAKLINE& line)
{
  int &level = line.brLevel;
  std::string next;
  bool toParams;
  if (level)
    {
      if (line.params.size())
	{
	  next = line.params[line.params.size()-1];
	  line.params.pop_back();
	  toParams = 1;
	} else
	{
	  next = line.cmd;
	  line.params.clear();
	  line.cmd.erase();
	  toParams = 0;
	}
    } else
    {
      line.cmd.erase();
      line.params.clear();
      toParams = 0;
    }
  int i;
  int state;
  if (level)
    state = 4; else
    state = 2;
  for(i=0;i<str.length();i++)
    {
      if (str[i] == ' ')
	{
	  if (state == 2)
	    continue;
	  if (state == 1)
	    {
	      state = 2;
	      continue;
	    }
	  if (state == 4)
	    {
	      next+=' ';
	      continue;
	    }
	  if (state == 0 || state == 5)
	    {
	      if (toParams)
		line.params.push_back(next); else
		  line.cmd = next;
	      next.erase();
	      toParams=1;
	      state = 1;
	      continue;
	    }
	  assert(0);
	} // spaces;
      if (str[i]=='{')
	{
	  if (state==1 || state == 2)
	    {
	      next.erase();
	      state = 4;
	      level=1;
	      continue;
	    }
	  if (state == 4)
	    {
	      next+=str[i];
	      level++;
	      continue;
	    }
	  if (state == 0)
	    {
	      next+=str[i]; 
	      continue;
	    }
	  continue;
	} // opening bracket;
      if (str[i] == '}')
	{
	  if (state == 4)
	    {
	      if (level==1)
		{
		  state=5;
		  level=0;
		} else
		  if (level > 1)
		    {
		      next+=str[i]; 
		      level--;
		    } else
		      {
			assert(0);
		      }
	      continue;
	    }
	  if (state == 0)
	    {
	      next+=str[i];
	      continue;
	    }
	  continue;
	} // closing bracket;
      // any char;
      if (state==0 || state==4)
	{
	  next+=str[i]; 
	  continue;
	}
      if (state == 1 || state == 2)
	{
	  next = str[i]; 
	  state = 0;
	}
    } // for
  if (state == 0 || state == 4 || state== 5)
    {
      if (toParams)
	line.params.push_back(next); else
	  line.cmd = next;
    }
}

static void handleEspeakLine(const ESPEAKLINE& line, std::string& queue)
{
  if (line.cmd == "l")
    {
      if (line.params.size()==0)
	return;
      if (line.params[0].empty())
	return;
      letter(line.params[0]);
      return;
    }
  if (line.cmd == "q")
    {
      if (line.params.size() == 0)
	return;
      queue += line.params[0];
      return;
    }
  if (line.cmd == "s")
    {
      stop();
      queue.erase();
      return;
    }
  if (line.cmd == "d")
    {
      removeDectalkCodes(queue);
      text(queue);
      queue.erase();
      return;
    }
  if (line.cmd == "tts_say")
    {
      if (line.params.size() == 0)
	return;
      std::string tt=line.params[0];
      removeDectalkCodes(tt);
      text(tt);
      return;
    }
  if (line.cmd == "t")
    {
      if (line.params.size() < 2)
	  return;
      if (!checkTypeUnsignedInt(line.params[0]))
	return;
      if (!checkTypeUnsignedInt(line.params[1]))
	return;
      tone(parseAsUnsignedInt(line.params[0]), parseAsUnsignedInt(line.params[1]));
      return;
    }
}

static void handleLine(const std::string& line)
{
  if (line.empty())
    return;
  static ESPEAKLINE el;
  static std::string queue;
  splitEspeakLine(line, el);
  if (el.brLevel)
    return;
  handleEspeakLine(el, queue);
}

static void readInput(int file)
{
  std::string line;
  while(1)
    {
      char buf[2048];
      int readCount;
      readCount = read(file, buf, sizeof(buf));
      if (readCount < 0)
	{
	  perror("read()");
	  return;
	}
      if (readCount==0)
	return;
      int i;
      for(i=0;i<readCount;i++)
	{
	  if (buf[i]==10)
	    {
	      handleLine(line);
	      line.erase();
	      continue;
	    }
	  if (buf[i] == 13)
	    continue;
	  line+=buf[i];
	}
    }
}

int main(int argc, char *argv[])
{
  for(int k=1;k<argc;k++)
    {
      std::string param=argv[k];
      if (trim(param)=="--help" || trim(param)=="-h")
	{
	  using namespace std;
	  cout << "VoiceMan client for Emacspeak." << endl;
	  cout << "Parameters:" << endl;
	  cout << "-h, --help - print this help screen;" << endl;
	  cout << "-t, --test - perform connection test." << endl;
	  return 0;
	}
      if (trim(param)=="--test" || trim(param)=="-t")
	{
	  std::cout << "Testing..." << std::endl;
	  tone(220, 150);
	  tone(220, 150);
	  tone(220, 150);
	  text("Performing client test...");
	  tone(220, 150);
	  tone(220, 150);
	  tone(220, 150);
	  return 0;
	}
    }
  readInput(0);
  vm_close(con);
  return 0;
}
