/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include<iostream>
#include<string>
#include<sstream>
#include<ext/hash_set>
#include<vector>
#include<assert.h>
#include<iconv.h>
#include"vmclient.h"
#include"transcoding.h"
#include"cmdline.h"
#include"strings.h"
#include"buildConfig.h"

#define PARAM_COUNT 4

static CMDARG cmdLineParams[PARAM_COUNT] = {
  {'h', "help", "", "show this help screen;"},
  {'s', "socket", "FILE_NAME", "connect to server via UNIX domain socket;"},
  {'p', "host", "HOST[:PORT]", "connect to server via TCP/IP;"},
  {'S', "say", "TEXT", "say TEXT and exit."}
};

static vm_connection_t con=VOICEMAN_BAD_CONNECTION;

#define VOLUME_HEAD "volume="
#define PITCH_HEAD "pitch="
#define RATE_HEAD "rate="

static bool checkBegins(const std::string& head, const std::string& str, std::string& tail)
{
  if (head.length() >= str.length())
    return 0;
  int i;
  for(i=0;i<head.length();i++)
    if (str[i] != head[i])
      return 0;
  tail.erase();
  for(i=head.length();i<str.length();i++)
    tail+=str[i];
  return 1;
}

static void process(const std::string& line)
{
  if (line.empty())
    {
      vm_stop(con);
      return;
    }
  if (line.length()==1)
    {
      std::wstring s=IO2WString(line);
      assert(s.length()==1);
      vm_letter(con, encodeUTF8(s).c_str());
      return;
    }
  std::string s;
  if (checkBegins(VOLUME_HEAD, line, s))
    {
      if (!checkTypeUnsignedInt(s))
	{
	  std::cerr << "ERROR:\'" << trim(s) << "\' is not a valid unsigned integer number." << std::endl;
	  return;
	}
      vm_volume(con, parseAsUnsignedInt(s));
      return;
    }
  if (checkBegins(PITCH_HEAD, line, s))
    {
      if (!checkTypeUnsignedInt(s))
	{
	  std::cerr << "ERROR:\'" << trim(s) << "\' is not a valid unsigned integer number." << std::endl;
	  return;
	}
      vm_pitch(con, parseAsUnsignedInt(s));
      return;
    }
  if (checkBegins(RATE_HEAD, line, s))
    {
      if (!checkTypeUnsignedInt(s))
	{
	  std::cerr << "ERROR:\'" << trim(s) << "\' is not a valid unsigned integer number." << std::endl;
	  return;
	}
      vm_rate(con, parseAsUnsignedInt(s));
      return;
    }
  vm_text(con, encodeUTF8(IO2WString(line)).c_str());
}

int main(int argc, char* argv[])
{
  using std::cout;
  using std::cin;
  using std::cerr;
  using std::endl;
  CmdArgsParser parser(cmdLineParams, PARAM_COUNT);
  if (!parser.parse(argc, argv))
    return 1;
  if (parser.used("help"))
    {
      cout << "Shell client for VOICEMAN speech system." << endl;
      cout << endl << "Command line options:" << endl;
      parser.printHelp();
      return 0;
    }
  if (parser.used("host"))
    {
      std::string param=parser["host"];
      std::string h=getDelimitedSubStr(param, 0, ':');
      std::string p=getDelimitedSubStr(param, 1, ':');
      if (trim(h).empty())
	h="localhost";
      int port;
      if (!trim(p).empty())
	{
	  if (!checkTypeUnsignedInt(p))
	    {
	      std::cerr << "ERROR:\'" << trim(p) << "\' is not a valid port number." << std::endl;
	      return 1;
	    }
	  port = parseAsUnsignedInt(p);
	} else 
	port=VOICEMAN_DEFAULT_PORT;
      con=vm_connect_inet(h.c_str(), port);
      if (con == VOICEMAN_BAD_CONNECTION)
	{
	  std::cerr << "ERROR:Could not establish connection to \'" << trim(h) << "\' to port " << port << "." << std::endl;
	  return 1;
	}
    } else
    if (parser.used("socket"))
      {
	std::string p=parser["socket"];
	if (trim(p).empty())
	  {
	    std::cerr << "ERROR:Missed name of UNIX domain socket." << std::endl;
	    return 1;
	  }
	if (trim(p) == "-")
	  p=VOICEMAN_DEFAULT_SOCKET;
	con = vm_connect_unix(p.c_str());
	if (con == VOICEMAN_BAD_CONNECTION)
	  {
	    std::cerr << "ERROR:Could not connect to server via UNIX domain socket \'" << p << "\'." << std::endl;
	    return 1;
	  }
      } else
      {
	con = vm_connect();
	if (con == VOICEMAN_BAD_CONNECTION)
	  {
	    cerr << "ERROR:Could not connect to voicemand with default settings." << endl;
	    return 1;
	  }
      }
  if (parser.used("say"))
    {
      vm_text(con, encodeUTF8(IO2WString(parser["say"])).c_str());
      vm_close(con);
      return 0;
    }
  cout << "VOICEMAN speech system. Ver: " << VOICEMAN_VERSION << "." << endl;
  cout << "Type \'quit\' or press Ctrl+D to leave this prompt." << endl;
  while(1)
    {
      std::string l;
      bool toQuit = 0;
      cout << "voiceman>";
      while(1)
	{
	  char c;
	  if (!cin.get(c))
	    {
	      toQuit = 1;
	      break;
	    }
	  if (c == 13)
	    continue;
	  if (c == 10)
	    break;
	  l+=c;
	}
      if (toQuit || (l == "quit"))
	break;
      process(l);
    }
  vm_close(con);
  cout << endl << "Bye!!!" << endl;
  return 0;
}
