/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"
#include"config.h"
#include"configuration.h"
#include"player.h"
#include"command.h"

void CommandPlayer::play(const TextItem& textItem)
{
  stop();
  if (m_pg) 
    while(waitpid(-1*m_pg, NULL, 0)>=0);
  std::string commandLine=prepareCommandLine(textItem.getVolume(), textItem.getPitch(), textItem.getRate());
  std::string text=prepareText(textItem);
  logMsg(LOG_TRACE, "Prepared command line is \'%s\'", commandLine.c_str());
  int pp[2];
  VM_SYS(pipe(pp)>=0, "pipe()");
  logMsg(LOG_TRACE, "Executing...");
  VM_SYS((m_pg = fork())>=0, "fork()");
  if (m_pg == 0)
    {
      setpgrp();
      close(pp[1]);
      int fd;
      VM_SYS1((fd = open(NULL_DEVICE, O_WRONLY))>=0, "open(NULL_DEVICE)");
      dup2(pp[0], STDIN_FILENO);
      dup2(fd, STDOUT_FILENO);
      dup2(fd, STDERR_FILENO);
      VM_SYS1(execlp("/bin/sh", "/bin/sh", "-c", commandLine.c_str(), NULL) != -1, "execlp()");
    } // new process;
  close(pp[0]);
  int r=write(pp[1], text.c_str(), text.length());
  if (r==-1)
    logMsg(LOG_ERROR, "Could not send prepared text to TTS. Failed write() call. (%s)", strerror(errno)); else
    if (r < text.length())
      logMsg(LOG_WARN, "Prepared text was sent to TTS uncompletely. (%d bytes instead of %d)", r, text.length());
  close(pp[1]);
}

void CommandPlayer::stop()
{
  if (m_pg == 0)
    return;
  killpg(m_pg, SIGINT);
  killpg(m_pg, SIGKILL);
}

bool CommandPlayer::processing() const
{
  if (m_pg == 0)
    return 0;
  pid_t p=waitpid(m_pg, NULL, WNOHANG);
  if (p > 0)
    while(waitpid(-1*m_pg, NULL, 0)>=0);
  return p==0;
}

void CommandPlayer::init(const OutputConfiguration& outputConfiguration)
{
  m_name=outputConfiguration.name;
  if (trim(outputConfiguration.command).empty())
    throw ConfigurationException("Command line for output \'"+m_name+"\' is not specified");
  m_cmdLine=trim(outputConfiguration.command);
  m_volumeFormat=outputConfiguration.volumeParams;
  m_pitchFormat=outputConfiguration.pitchParams;
  m_rateFormat=outputConfiguration.rateParams;
  logMsg(LOG_TRACE, "Loaded %d cap replacements for output \'%s\'", outputConfiguration.capReplacements.size(), getName().c_str());
  m_capList=outputConfiguration.capReplacements;
}

std::string CommandPlayer::prepareFloatValue(TextParam value, const FloatValueParams& format) const
{
  assert(format.digits>=0 && format.digits<= 10);
  float floatValue=value.getValue(format.min, format.max);
  return makeStringFromDouble<std::string>(floatValue, format.digits);
}

std::string CommandPlayer::prepareCommandLine(TextParam volume, TextParam pitch, TextParam rate) const
{
  std::string volumeStr, pitchStr, rateStr;
  volumeStr=prepareFloatValue(volume, m_volumeFormat);
  pitchStr=prepareFloatValue(pitch, m_pitchFormat);
  rateStr = prepareFloatValue(rate, m_rateFormat);
  std::string s;
  for(int i=0;i<m_cmdLine.length();i++)
    {
      if (m_cmdLine[i] != '%' || i+1>=m_cmdLine.length())
	{
	  s+=m_cmdLine[i];
	  continue;
	}
      i++;
      switch(m_cmdLine[i])
	{
	case 'v':
	  s+=volumeStr;
	  break;
	case 'p':
	  s+=pitchStr;
	  break;
	case 'r':
	  s+=rateStr;
	  break;
	default:
	  s+=m_cmdLine[i];
	}
    }
  return s;
}

std::wstring CommandPlayer::makeCaps(const TextItem& textItem) const
{
  std::wstring oldText=textItem.getText();
  std::wstring text;
  const Lang* lang=textItem.getLang();
  for(int i=0;i<oldText.length();i++)
    {
      if (textItem.isMarked(i))
	{
	  wchar_t c=oldText[i];
	  if (lang)
	    c=lang->toLower(c);
	  attachSpace(text);
	  std::map<wchar_t, std::wstring>::const_iterator it=m_capList.find(c);
	  if (it!=m_capList.end())
	    attachStringWithSpace(text, it->second); else
	      attachCharWithSpace(text, oldText[i]);
	} else
	  attachCharWithoutDoubleSpaces(text, oldText[i]);
    }
  return trim(text);
}

std::string CommandPlayer::prepareText(const TextItem& textItem) const
{
  std::wstring text=makeCaps(textItem);
  std::string s = encodeUTF8(text);
  logMsg(LOG_TRACE, "Prepared text is \'%s\'", s.c_str());
  s += '\n';
  return s;
}
