/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"
#include"config.h"

std::string readTextFile(const std::string& fileName);

void ConfigFile::load(const std::string& fileName)
{
  std::string s=readTextFile(fileName);
  std::vector<std::string> lines;
  int i;
  std::string ss;
  for(i=0;i<s.length();i++)
    {
      if (s[i]==13)
	continue;
      if (s[i]==10)
	{
	  lines.push_back(ss);
	  ss.erase();
	  continue;
	}
      ss += s[i];
    }// lines splitting;
  lines.push_back(ss);
  ConfigFileSection sec;
  bool secInited = 0;
  for(i=0;i<lines.size();i++)
    {
      std::string s1, s2;
      int typeCode;
      int state=process(lines[i], typeCode, s1, s2);
      if (state!=0)
	throw ConfigFileException(state, fileName, i+1);
      if (typeCode==0)
	continue;
      if (typeCode==1)
	{
	  if (secInited)
	    m_sections.push_back(sec);
	  secInited=1;
	  if (s1.empty())
	    throw ConfigFileException(ConfigFileException::EMPTY_SECTION_NAME, fileName, i+1);
	  sec.setName(s1);
	  sec.clear();
	  continue;
	} // new section;
      if (typeCode == 2)
	{
	  if (!secInited)
	    throw ConfigFileException(ConfigFileException::PARAMETER_WITHOUT_SECTION, fileName, i+1);
	  sec.add(trim(s1), s2);
	  continue;
	} // param;
      assert(0);
    } // for lines;
  if (secInited)
    m_sections.push_back(sec);
}

int ConfigFile::process(const std::string& line, int& code, std::string& str1, std::string& str2) const
{
  const std::string loLine=toLower(line);
  int i;
  int state=0;
  str1.erase();
  str2.erase();
  code=0;
  for(i=0;i<line.length();i++)
    {
      if (state==0 && BLANK_CHAR(line[i]))
	continue;
      if (state==0 && line[i]=='#')
	return 0;
      if (state==0 && line[i]=='[')
	{
	  code=1;
	  for(i++;i<line.length();i++)
	    {
	      if (BLANK_CHAR(line[i]))
		continue;
	      if (line[i]==']')
		return 0;
	      if (loLine[i]>='a' && loLine[i]<='z')
		{
		  str1+=loLine[i];
		  continue;
		}
	      return ConfigFileException::INVALID_SECTION_HEADER;
	    }  // for;
	} // section;
      if (state==0 && loLine[i]>='a' && loLine[i]<='z')
	{
	  str1+=loLine[i];
	  code=2;
	  state=1;
	  continue;
	}
      if (state==1 && (BLANK_CHAR(line[i]) || line[i]=='_'))
	continue;
      if (state==1 && loLine[i]>='a' && loLine[i]<='z')
	{
	  str1+=loLine[i];
	  continue;
	}
      if (state==1 && DIGIT_CHAR(line[i]))
	{
	  str1+=line[i];
	  continue;
	}
      if (state==1 && line[i]=='=')
	{
	  state=2;
	  continue;
	}
      if (state==2 && BLANK_CHAR(line[i]))
	continue;
      if (state == 2 && ((loLine[i]>='a' && loLine[i] <= 'z') || DIGIT_CHAR(line[i])))
	{
	  for(;i<line.length();i++)
	    {
	      if (line[i]=='_' || (loLine[i]>='a' && loLine[i] <= 'z') || DIGIT_CHAR(line[i]))
		str2+=line[i]; else
		  break;
	    } // for;
	  for(;i<line.length();i++)
	    {
	      if (BLANK_CHAR(line[i]))
		continue;
	      if (line[i]=='#')
		return 0;
	      return ConfigFileException::INVALID_UNQUOTED_VALUE;
	    }
	  return 0;
	} // no quotas;
      if (state==2 && line[i]=='\"')
	{
	  state=3;
	  continue;
	}
      if (state==3 && line[i]!='\"')
	{
	  str2+=line[i];
	  continue;
	}
      if (state==3 && line[i]=='\"')
	{
	  state=4;
	  continue;
	}
      if (state==4 && line[i]=='\"')
	{
	  state=3;
	  str2+='\"';
	  continue;
	}
      if (state==4 && line[i]!='\"')
	return 0;
      switch(state)
	{
	case 0:
	  return ConfigFileException::INVALID_LINE_BEGINNING;
	case 1:
	  return ConfigFileException::INVALID_PARAMETER_NAME;
	case 2:
	  return ConfigFileException::INVALID_CHAR_AFTER_EQUALS;
	default:
	  assert(0);
	}
    } // for;
  assert(state!=0 || code==0);
  if (state==4 || state==0)
    return 0;
  return ConfigFileException::UNEXPECTED_LINE_END;;
}

int ConfigFile::params(const std::string& section, const std::string& param) const
{
  return ANY_VALUE;
}

void ConfigFile::checkParamsInSection(const ConfigFileSection& section) const
{
  for(ConfigFileSection::const_iterator it=section.begin();it!=section.end();it++)
    {
      int requiredType=params(section.getName(), it->first);
      const std::string& value=it->second;
      if (requiredType==ANY_VALUE)
	continue;
      if (requiredType==INVALID_VALUE)
	throw ConfigFileValueTypeException("unknown parameter \'"+it->first+"\' in section \'"+section.getName()+"\'");
      if (requiredType==BOOLEAN_VALUE && !checkTypeBool(value))
	throw ConfigFileValueTypeException("parameter \'"+it->first+"\' in section \'"+section.getName()+"\' should be boolean. value \'"+trim(value)+"\' is illegal");
      if (requiredType==STRING_VALUE && trim(value).empty())
	throw ConfigFileValueTypeException("parameter \'"+it->first+"\' in section \'"+section.getName()+"\' cannot have an empty value");
      if (requiredType==INT_VALUE && !checkTypeInt(value))
	throw ConfigFileValueTypeException("parameter \'"+it->first+"\' in section \'"+section.getName()+"\' should be an integer number. Value \'"+trim(value)+"\' is illegal");
      if (requiredType==UINT_VALUE && !checkTypeUnsignedInt(value))
	throw ConfigFileValueTypeException("parameter \'"+it->first+"\' in section \'"+section.getName()+"\' should be an unsigned integer number. Value \'"+trim(value)+"\' is illegal");
    }
}

void ConfigFile::checkParams() const
{
  for(int i=0;i<m_sections.size();i++)
    checkParamsInSection(m_sections[i]);
}
