/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#ifndef __VOICEMAN_CONFIG_H__
#define __VOICEMAN_CONFIG_H__

#define VOICEMAN_BEGIN_PARAM_TABLE(cl) int cl::params(const std::string& section, const std::string& param) const {
#define VOICEMAN_DECLARE_PARAM(s,p) if (section==(s) && param==(p)) return ANY_VALUE
#define VOICEMAN_DECLARE_BOOLEAN_PARAM(s,p) if (section==(s) && param==(p)) return BOOLEAN_VALUE
#define VOICEMAN_DECLARE_STRING_PARAM(s,p) if (section==(s) && param==(p)) return STRING_VALUE
#define VOICEMAN_DECLARE_INT_PARAM(s,p) if (section==(s) && param==(p)) return INT_VALUE
#define VOICEMAN_DECLARE_UINT_PARAM(s,p) if (section==(s) && param==(p)) return UINT_VALUE
#define VOICEMAN_END_PARAM_TABLE return INVALID_VALUE;}

class ConfigFileException
{
public:
  enum {INVALID_SECTION_HEADER=1, EMPTY_SECTION_NAME=2, PARAMETER_WITHOUT_SECTION=3, INVALID_UNQUOTED_VALUE=4, INVALID_LINE_BEGINNING=5, INVALID_PARAMETER_NAME=6, INVALID_CHAR_AFTER_EQUALS=7, UNEXPECTED_LINE_END=8};

  ConfigFileException(int code): m_code(code), m_line(-1) {}
  ConfigFileException(int code, const std::string& fileName, int line): m_code(code), m_fileName(fileName), m_line(line) {}
  virtual ~ConfigFileException() {}

  int getCode() const 
  {
    return m_code;
  }

  int getLine() const
  {
    return m_line;
  }

  std::string getFileName() const
  {
    return m_fileName;
  }

  std::string getDescr() const
  {
    switch(m_code)
      {
      case INVALID_SECTION_HEADER:
	return "invalid section header format";
      case EMPTY_SECTION_NAME:
	return "missed section name";
      case PARAMETER_WITHOUT_SECTION:
	return "section is not opened";
      case INVALID_UNQUOTED_VALUE:
	return "invalid char in value without quotas";
      case INVALID_LINE_BEGINNING:
	return "invalid char before parameter name";
      case INVALID_PARAMETER_NAME:
	return "invalid char in parameter name";
      case INVALID_CHAR_AFTER_EQUALS:
	return "invalid char after \'=\' sign";
      case UNEXPECTED_LINE_END:
	return "unexpected line end";
      default:
	assert(0);
      }
  }

  std::string getMessage() const 
  {
    if (m_line<=0)
      return getDescr();
    std::ostringstream s;
    s << m_fileName << "(" << m_line << "):" << getDescr();
    return s.str();
  }

private:
  std::string m_fileName;
  int m_code, m_line;
}; // class ConfigFileException;

class ConfigFileValueTypeException
{
public:
  ConfigFileValueTypeException(const std::string& descr): m_descr(descr) {}
  virtual ~ConfigFileValueTypeException() {}

  std::string getMessage() const
  {
    return m_descr;
  }

private:
  std::string m_descr;
}; //class ConfigFileValueTypeException;

class ConfigFileSection: public std::map<std::string, std::string>
{
public:
  ConfigFileSection() {}
  ConfigFileSection(const std::string& name): m_name(name) {}
  virtual ~ConfigFileSection() {}

  std::string getName() const 
  {
    return m_name;
  }

  void setName(const std::string& name)
  {
    m_name=name;
  }

  bool has(const std::string& name) const 
  {
    return find(name)!=end();
  }

  void add(const std::string& param, const std::string& value)
  {
    insert(value_type(param, value));
  }

  const std::string& operator [](const std::string& name) const
  {
    const_iterator it=find(name);
    assert(it!=end());
    return it->second;
  }

private:
  std::string m_name;
}; // class ConfigFileSection;

class ConfigFile
{
public:
  ConfigFile() {}
  virtual ~ConfigFile() {}

  void load(const std::string& fileName);
  void checkParams() const;

  int getSectionCount() const 
  {
    return m_sections.size();
  }

  const ConfigFileSection& getSection(int index) const
  {
    assert(index <= m_sections.size());
    return m_sections[index];
  }

  const ConfigFileSection& findSection(const std::string& name) const
  {
    for(int i=0;i<m_sections.size();i++)
      if (m_sections[i].getName()==toLower(trim(name)))
	return m_sections[i];
    assert(0);
  }

  bool hasSection(const std::string& name) const
  {
    for(int i=0;i<m_sections.size();i++)
      if (m_sections[i].getName()==toLower(trim(name)))
	return 1;
    return 0;
  }

protected:
  enum {ANY_VALUE = -1, INVALID_VALUE = 0, BOOLEAN_VALUE = 1, STRING_VALUE = 2, INT_VALUE = 3, UINT_VALUE = 4};
  virtual int params(const std::string& section, const std::string& param) const = 0;

private:
  int process(const std::string& line, int& code, std::string& str1, std::string& str2) const;
  void checkParamsInSection(const ConfigFileSection& section) const;


private:
  std::vector<ConfigFileSection> m_sections;
}; //class ConfigFile;

#endif // __VOICEMAN_CONFIG_H__
