/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"
#include"config.h"
#include"configuration.h"

VOICEMAN_BEGIN_PARAM_TABLE(VoicemanConfigFile)

VOICEMAN_DECLARE_UINT_PARAM("global", "defaultvolume");
VOICEMAN_DECLARE_UINT_PARAM("global", "defaultpitch");
VOICEMAN_DECLARE_UINT_PARAM("global", "defaultrate");
VOICEMAN_DECLARE_PARAM("global", "startupmessage");
VOICEMAN_DECLARE_STRING_PARAM("global", "datadir");
VOICEMAN_DECLARE_STRING_PARAM("global", "digits");
VOICEMAN_DECLARE_BOOLEAN_PARAM("global", "capitalization");
VOICEMAN_DECLARE_BOOLEAN_PARAM("global", "separation");
VOICEMAN_DECLARE_BOOLEAN_PARAM("global", "tones");
VOICEMAN_DECLARE_BOOLEAN_PARAM("global", "tonesinqueue");
VOICEMAN_DECLARE_STRING_PARAM("global", "socket");
VOICEMAN_DECLARE_UINT_PARAM("global", "port");
VOICEMAN_DECLARE_UINT_PARAM("global", "loglevel");
VOICEMAN_DECLARE_STRING_PARAM("global", "logfilename");
VOICEMAN_DECLARE_BOOLEAN_PARAM("global", "logtostderr");
VOICEMAN_DECLARE_UINT_PARAM("global", "maxqueue");
VOICEMAN_DECLARE_UINT_PARAM("global", "loopdelay");
VOICEMAN_DECLARE_UINT_PARAM("global", "maxinputline");
VOICEMAN_DECLARE_UINT_PARAM("global", "maxclients");

VOICEMAN_DECLARE_STRING_PARAM("output", "type");
VOICEMAN_DECLARE_STRING_PARAM("output", "name");
VOICEMAN_DECLARE_STRING_PARAM("output", "lang");
VOICEMAN_DECLARE_STRING_PARAM("output", "command");
VOICEMAN_DECLARE_STRING_PARAM("output", "pitch");
VOICEMAN_DECLARE_STRING_PARAM("output", "rate");
VOICEMAN_DECLARE_STRING_PARAM("output", "volume");
VOICEMAN_DECLARE_STRING_PARAM("output", "caplist");

VOICEMAN_DECLARE_STRING_PARAM("default", "output");
VOICEMAN_DECLARE_STRING_PARAM("default", "chars");

VOICEMAN_END_PARAM_TABLE

static std::string makeBooleanValue(bool value)
{
  if (value)
    return "YES";
  return "NO";
}

void OutputConfiguration::init(const ConfigFileSection& section)
{
  assert(section.getName()=="output");
  if (!section.has("name"))
    throw ConfigurationException("There is the output in your configuration file without name.");
  name=trim(section["name"]);
  if (!section.has("type"))
    throw ConfigurationException("Output \'"+name+"\' has no type information");
  std::string typeStr=section["type"];
  typeStr=toLower(trim(typeStr));
  if (typeStr=="command")
    type=COMMAND; else
      throw ConfigurationException("Output \'"+name+"\' has the invalid value for \'type\' parameter \'"+typeStr+"\'");
  if (section.has("lang"))
    lang=toLower(trim(section["lang"]));
  if (section.has("command"))
    command=trim(section["command"]);
  if (section.has("volume"))
    volumeParams.init(trim(section["volume"]), name, "volume");
  if (section.has("pitch"))
    pitchParams.init(trim(section["pitch"]), name, "pitch");
  if (section.has("rate"))
    rateParams.init(trim(section["rate"]), name, "rate");
  if (section.has("caplist"))
    {
      std::wstring capList; 
      if (!decodeUTF8(section["caplist"], capList))
	throw UTF8Exception("String for parameter \'cap list\' for output \'"+name+"\' contains illegal UTF-8 sequence");
      capList=trim(capList);
      StringDelimitedIterator<std::wstring> it(capList, L" ");
      while(it.next())
	{
	  std::wstring s1=it.str();
	  if (!it.next())
	    throw ConfigurationException("Output \'"+name+"\' contains uncomplete pair in cap list");
	  std::wstring s2=it.str();
	  if (s1.length() != 1)
	    throw ConfigurationException("Output \'"+name+"\' has illegal pair in cap list. First item should contain only one character.");
      capReplacements[s1[0]]=s2;
	}
    }
}

void FloatValueParams::init(const std::string& value, const std::string& outputName, const std::string& name)
{
  std::string digitsStr, minStr, maxStr;
  digitsStr=trim(getDelimitedSubStr(value, 0, ':'));
  minStr=trim(getDelimitedSubStr(value, 1, ':'));
  maxStr=trim(getDelimitedSubStr(value, 2, ':'));
  if (!digitsStr.empty())
    {
      if (!checkTypeUnsignedInt(digitsStr))
	throw ConfigurationException("Number of digits specified for "+name+" for output \'"+outputName+"\' is not an unsigned integer number");
      digits=parseAsUnsignedInt(digitsStr);
      if (digits> 10)
	throw ConfigurationException("Number of digits specified for "+name+" for output \'"+outputName+"\' should not be grater than 10");
    }
  if (!minStr.empty())
    {
      if (!checkTypeDouble(minStr))
	throw ConfigurationException("Min value specified for "+name+" for output \'"+outputName+"\' is not a valid double number");
      min=parseAsDouble(minStr);
    }
  if (!maxStr.empty())
    {
      if (!checkTypeDouble(maxStr))
	throw ConfigurationException("Max value specified for "+name+" for output \'"+outputName+"\' is not a valid double number");
      max=parseAsDouble(maxStr);
    }
  if (digits < 0 || digits>10)
    throw ConfigurationException("Number of digits for "+name+" for output \'"+outputName+"\' should be between 0 and 10");
}

void Configuration::printConfiguration() const
{
  std::cout << "*** GENERAL: ***" << std::endl;
  std::cout << "Start up message: " << startUpMessage << std::endl;
  std::cout << "Default output name: " << defaultOutputName << std::endl;
  std::cout << "Default output chars: " << defaultOutputChars << std::endl;
  std::cout << "Log file name: " << logFileName << std::endl;
  std::cout << "Log level: " << logLevel << std::endl;
  std::cout << "Log to stderr: " << makeBooleanValue(logToStdErr) << std::endl;
  std::cout << "Pid file name: " << pidFileName << std::endl;
  std::cout << "Data directory: " << dataDir << std::endl;
  std::cout << "String to say: " << toSay << std::endl;
  std::cout << "Daemon: " << makeBooleanValue(daemon) << std::endl;
  std::cout << "Test configuration: " << makeBooleanValue(testConfiguration) << std::endl;
  std::cout << "Max queue size: " << maxQueue << std::endl;
  std::cout << "Loop delay: " << loopDelay << std::endl;
  std::cout << "Unix socket: " << unixSocket << std::endl;
  std::cout << "Use TCP/IP socket: " << makeBooleanValue(useInetSocket) << std::endl;
  if (useInetSocket)
    std::cout << "Port for TCP/IP socket: " << inetPort << std::endl;
  std::cout << "Default volume: " << defaultVolume.getValue() << std::endl;
  std::cout << "Default pitch: " << defaultPitch.getValue() << std::endl;
  std::cout << "Default rate: " << defaultRate.getValue() << std::endl;
  std::cout << "Max clients: " << maxClients << std::endl;
  std::cout << "Max line length: " << maxLine << std::endl;
  std::cout << "Digits mode: ";
  switch(digitsMode)
    {
    case DIGITS_NORMAL:
      std::cout << "NORMAL" << std::endl;
      break;
    case DIGITS_SINGLE:
      std::cout << "SINGLE" << std::endl;
      break;
    case DIGITS_NONE:
      std::cout << "NONE" << std::endl;
      break;
    default:
      assert(0);
    }
  std::cout << "Tones: " << makeBooleanValue(tones) << std::endl;
  std::cout << "Tones in queue: " << makeBooleanValue(tonesInQueue) << std::endl;
  std::cout << "Capitalization: " << makeBooleanValue(capitalization) << std::endl;
  std::cout << "Separation: " << makeBooleanValue(separation) << std::endl;
  std::cout << std::endl << "*** OUTPUTS: ***" << std::endl;
  for(int i=0;i<outputs.size();i++)
    {
      outputs[i].printConfiguration();
      std::cout << std::endl;
    }
}

void OutputConfiguration::printConfiguration() const
{
  std::cout << "Type: ";
  switch(type)
    {
    case COMMAND:
      std::cout << "COMMAND" << std::endl;
      break;
    default:
      assert(0);
    }
  std::cout << "Name: " << name << std::endl;
  std::cout << "Language: " << lang << std::endl;
  std::cout << "Command: " << command << std::endl;
  std::cout << "Digits in volume value: " << volumeParams.digits << std::endl;
  std::cout << "Min volume value: " << volumeParams.min << std::endl;
  std::cout << "Max volume value: " << volumeParams.max << std::endl;
  std::cout << "Digits in pitch value: " << pitchParams.digits << std::endl;
  std::cout << "Min pitch value: " << pitchParams.min << std::endl;
  std::cout << "Max pitch value: " << pitchParams.max << std::endl;
  std::cout << "Digits in rate value: " << rateParams.digits << std::endl;
  std::cout << "Min rate value: " << rateParams.min << std::endl;
  std::cout << "Max rate value: " << rateParams.max << std::endl;
  std::cout << "Capitals: " << std::endl;
  std::map<wchar_t, std::wstring>::const_iterator it;
  for (it=capReplacements.begin();it!=capReplacements.end();it++)
    {
      std::wstring s;
      s+=it->first;
      std::cout << s << "=(" << it->second << ") ";
    }
  std::cout << std::endl;
}
