/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"
#include"config.h"
#include"configuration.h"
#include"sockets.h"
#include"clients.h"
#include"protocol.h"
#include"player.h"
#include"command.h"
#include"queue.h"
#include"tone.h"
#include"languages.h"
#include"dispatcher.h"

#define MIN_TONE_FREQ 50
#define MAX_TONE_FREQ 5000

#define MIN_TONE_LENGTH 10
#define MAX_TONE_LENGTH 10000

void Dispatcher::text(const Client& client, const std::wstring& t)
{
  TextItem item(t, client.getVolume(), client.getPitch(), client.getRate());
  std::list<TextItem> items;
  m_proc->process(item, items);
  std::list<TextItem>::iterator it;
  for(it=items.begin();it!=items.end();it++)

    tqAddText(*it);
}

void Dispatcher::letter(const Client& client, wchar_t c)
{
  std::list<TextItem> items;
  m_proc->processLetter(c, client.getVolume(), client.getPitch(), client.getRate(), items);
  tqStop();
  std::list<TextItem>::iterator it;
  for(it=items.begin();it!=items.end();it++)
    tqAddText(*it);
}

void Dispatcher::stop(const Client& client)
{
  tqStop();
}

void Dispatcher::tone(const Client& client, int freq, int lengthMs)
{
  if (!m_params.tones)
    return;
  if (lengthMs < MIN_TONE_LENGTH || lengthMs> MAX_TONE_LENGTH)
    {
      logMsg(LOG_WARN, "Rejecting tone with illegal length. (%dms)", lengthMs);
      return; 
    }
  if (freq < MIN_TONE_FREQ || freq > MAX_TONE_FREQ)
    {
      logMsg(LOG_WARN, "Rejecting tone with illegal frequency (%dHz).", freq);
      return;
}
  if (!m_params.tonesInQueue)
    {
      TonePlayer tonePlayer;
      tonePlayer.play(freq, lengthMs);
    } else
      tqAddTone(freq, lengthMs);
}

void Dispatcher::param(Client& client, int paramType, TextParam value)
{
  switch(paramType)
    {
    case PARAM_VOLUME:
      client.setVolume(value);
      break;
    case PARAM_PITCH:
      client.setPitch(value);
      break;
    case PARAM_RATE:
      client.setRate(value);
      break;
    default:
      assert(0);
    }
}

void Dispatcher::initTextProcessor(const Configuration& configuration)
{
  int i;
  const std::vector<OutputConfiguration>& outputs=configuration.outputs;
  DelimitedFile charsTable, replacements;
  charsTable.read(configuration.dataDir+"charstable");
  replacements.read(configuration.dataDir+"replacements");
  logMsg(LOG_TRACE, "%d special values were loaded.", charsTable.getLineCount());
  logMsg(LOG_TRACE, "%d replacements were loaded.", replacements.getLineCount());
  for(i=0;i<charsTable.getLineCount();i++)
    {
      std::wstring ch, value;
      if (!decodeUTF8(charsTable.getItem(i, 0), ch) || !decodeUTF8(charsTable.getItem(i, 1), value))
	throw UTF8Exception("line \'"+charsTable.getRawLine(i)+"\' in your chars table contains illegal UTF-8 sequence");
      ch=trim(ch);
      value=trim(value);
      wchar_t c;
      if (ch.length()==1)
	c=ch[0]; else
	  if (toLower(ch)==L"cln")
	    c=L':'; else
	      if (toLower(ch)==L"hsh")
		c=L'#'; else
		  throw ConfigurationException("Each line in your chars table file should contain as first item the single char or \'cln\' string or \'hsh\' string. Line \'"+charsTable.getRawLine(i)+"\' is illegal.");
      if (value.empty())
	throw ConfigurationException("Second item in your chars table file should not be empty. Line \'"+charsTable.getRawLine(i)+"\' is illegal.");
      m_proc->setSpecialValueFor(c, value);
    }
  for(i=0;i<replacements.getLineCount();i++)
    {
      std::string outputName;
      std::wstring oldValue, newValue;
      outputName=replacements.getItem(i, 0);
      if (!decodeUTF8(replacements.getItem(i, 1), oldValue) || !decodeUTF8(replacements.getItem(i, 2), newValue))
	throw UTF8Exception("line \'"+replacements.getRawLine(i)+"\' in your replacements file contain illegal UTF-8 sequence");
      outputName=trim(outputName);
      oldValue=trim(oldValue);
      newValue=trim(newValue);
      if (outputName.empty())
	throw ConfigurationException("The first item in your replacements file should not be empty. Line \'"+replacements.getRawLine(i)+"\' is illegal.");
      if (oldValue.empty())
	throw ConfigurationException("The second item in your replacements file should not be empty. Line \'"+replacements.getRawLine(i)+"\' is illegal");
      if (toLower(oldValue)==L"cln")
	oldValue=L":";
      if (toLower(oldValue)==L"hsh")
	oldValue=L"#";
      m_proc->addReplacement(outputName, oldValue, newValue);
    }
  for(i=0;i<outputs.size();i++)
    {
      const std::string& name=outputs[i].name;
      const Lang* lang=languages.getLang(outputs[i].lang);
      if (lang == NULL)
	logMsg(LOG_WARN, "Output \'%s\' contain invalid language information \'%s\'", name.c_str(), outputs[i].lang.c_str()); else
	  m_proc->associate(lang->getAllChars(), name);
      m_proc->setLang(name, lang);
    }
  m_proc->associate(configuration.defaultOutputChars, DEFAULT_OUTPUT_NAME);
  m_proc->setDefaultOutput(configuration.defaultOutputName);
}

std::wstring Dispatcher::splitAndProcess(const std::wstring& data, Client& client)
{
  std::wstring s;
  int i;
  for(i=0;i<data.length();i++)
    {
      if (data[i]=='\n')
	{
	  if (!s.empty() && !client.isRejecting())
	    line(s, client);
	  s.erase();
	  client.setRejectingState(0);
	  continue;
	}
      if (m_params.maxLine != 0 && s.length() > m_params.maxLine)
	{
	  if (!client.isRejecting())
	    logMsg(LOG_WARN, "Received string exceeds input line length limit. Rejecting extra characters.");;
	  client.setRejectingState(1);
	}
      if (!client.isRejecting())
	s+=data[i];
    }
  return s;
}

void Dispatcher::handleData(const std::wstring& data, Client& client)
{
  std::wstring s;
  int i;
  for(i=0;i<data.length();i++)
    if (data[i]!=13) 
      s += data[i];
  std::wstring toProcess=client.getChain()+s;
  client.setChain(splitAndProcess(toProcess, client));
}
