/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#ifndef __VOICEMAN_EXCEPTION_H__
#define __VOICEMAN_EXCEPTION_H__

class Exception
{
public:
  Exception() {}
  virtual ~Exception() {}

  virtual std::string getMessage() const = 0;
  virtual void makeLogReport() const = 0;
}; //class Exceptin;

class SystemException: public Exception
{
public:
  SystemException(): m_code(errno) {}
  SystemException(const std::string& comment): m_code(errno), m_comment(comment) {}
  SystemException(int code): m_code(code) {}
  SystemException(int code, const std::string& comment): m_code(code), m_comment(comment) {}
  virtual ~SystemException() {}

  int getCode() const
  {
    return m_code;
  }

  std::string getDescr() const
  {
    return strerror(m_code);
  }

  std::string getComment() const
  {
    return m_comment;
  }

  std::string getMessage() const
  {
    return getComment()+":"+getDescr();
  }

  void makeLogReport() const
  {
    logMsg(LOG_ERROR, "System call error:%s", getMessage().c_str());
  }

private:
  int m_code;
  std::string m_comment;
}; //class SystemException;

class ConfigurationException: public Exception
{
public:
  ConfigurationException(const std::string& comment): m_comment(comment) {}
  virtual ~ConfigurationException() {}

  std::string getMessage() const
  {
    return m_comment;
  }

  void makeLogReport() const
  {
    logMsg(LOG_ERROR, "Configuration error:%s", getMessage().c_str());
  }

private:
  std::string m_comment;
}; //class ConfigurationException;

class UTF8Exception: public Exception
{
public:
  UTF8Exception(const std::string& comment): m_comment(comment) {}
  virtual ~UTF8Exception() {}

  std::string getMessage() const
  {
    return m_comment;
  }

  void makeLogReport() const
  {
    logMsg(LOG_ERROR, "UTF-8 processing error:%s", m_comment.c_str());
  }

private:
  std::string m_comment;
}; //class UTF8Exception;

#define VM_SYS(expr, msg) if (!(expr)) throw SystemException(msg)

#endif //__VOICEMAN_EXCEPTION_H__
