/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"
#include"config.h"
#include"configuration.h"
#include"sockets.h"
#include"clients.h"
#include"player.h"
#include"command.h"
#include"queue.h"
#include"protocol.h"
#include"dispatcher.h"
#include"cmdline.h"
#include"languages.h"

static Configuration configuration;
static std::list<Socket*> toListen;

#define PARAM_COUNT 6
static CMDARG cmdLineParams[PARAM_COUNT] = {
  {'h', "help", "", "show this help screen;"},
  {'d', "daemon", "", "run the server in daemon mode;"},
  {'c', "config", "fILE_NAME", "Use specified configuration file;"},
  {'S', "say", "TEXT", "say TEXT and exit;"},
  {'p', "pidfile", "FILE_NAME", "put pid of server process to FILE_NAME;"},
  {'t', "test", "", "only load and show configuration information."}
};

static void printLogo(bool shortly)
{
  std::cout << "VoiceMan speech processing system. Version: " << VOICEMAN_VERSION << "." << std::endl;
  if (!shortly)
    std::cout << "Press Ctrl+C to terminate this server." << std::endl << std::endl;
}

static void printHelp()
{
  printLogo(1);
  std::cout << "Command line arguments:" << std::endl;
  CmdArgsParser parser(cmdLineParams, PARAM_COUNT);
  parser.printHelp();
}

static void loadConfiguration(int argc, char *argv[])
{
  CmdArgsParser cmdLine(cmdLineParams, PARAM_COUNT);
  if (!cmdLine.parse(argc, argv))
    exit(1);
  if (cmdLine.used("help"))
    {
      printHelp();
      exit(0);
    }
  std::string configFileName;
  if (cmdLine.used("config"))
    configFileName=cmdLine["config"]; else
      configFileName=VOICEMAN_DEFAULT_CONFIG;
  VoicemanConfigFile configFile;
  configFile.load(trim(configFileName));
  configFile.checkParams();
  if (cmdLine.used("say"))
    {
      if (!decodeUTF8(cmdLine["say"], configuration.toSay))
	throw UTF8Exception("Text to be spoken contains an illegal UTF-8 sequence.");
    }
  if (cmdLine.used("pidfile"))
    configuration.pidFileName=cmdLine["pidfile"];
  configuration.daemon=cmdLine.used("daemon");
  configuration.testConfiguration=cmdLine.used("test");
  if (configFile.hasSection("global"))
    {
      const ConfigFileSection& globalSection=configFile.findSection("global");
      if (globalSection.has("socket"))
	configuration.unixSocket=trim(globalSection["socket"]);
      if (globalSection.has("port"))
	{
	  configuration.useInetSocket=1;
	  configuration.inetPort=parseAsUnsignedInt(globalSection["port"]);
	}
      if (globalSection.has("defaultvolume"))
	configuration.defaultVolume=parseAsUnsignedInt(globalSection["defaultvolume"]);
      if (globalSection.has("defaultpitch"))
	configuration.defaultPitch=parseAsUnsignedInt(globalSection["defaultpitch"]);
      if (globalSection.has("defaultrate"))
	configuration.defaultRate=parseAsUnsignedInt(globalSection["defaultrate"]);
      if (globalSection.has("digits"))
	{
	  std::string value=toLower(trim(globalSection["digits"]));
	  if (value== "normal")
	    configuration.digitsMode=DIGITS_NORMAL;  else
	      if (value=="none")
		configuration.digitsMode=DIGITS_NONE; else
		  if (value=="single")
		    configuration.digitsMode=DIGITS_SINGLE; else 
		      throw ConfigFileValueTypeException("Parameter \'digits\' in section \'global\' has an invalid value \'"+value+"\'. Allowed values are \'normal\', \'none\' and \'single\'.");
	}
      if (globalSection.has("tones"))
	configuration.tones=parseAsBool(globalSection["tones"]);
      if (globalSection.has("tonesinqueue"))
	configuration.tonesInQueue=parseAsBool(globalSection["tonesinqueue"]);
      if (globalSection.has("startupmessage"))
	{
	  if (!decodeUTF8(globalSection["startupmessage"], configuration.startUpMessage))
	    throw UTF8Exception("Value for parameter \'start up message\' in your configuration file contains an illegal UTF-8 sequence.");
	}
      if (globalSection.has("capitalization"))
	configuration.capitalization=parseAsBool(globalSection["capitalization"]);
      if (globalSection.has("separation"))
	configuration.separation=parseAsBool(globalSection["separation"]);
      if (globalSection.has("maxclients"))
	configuration.maxClients=parseAsUnsignedInt(globalSection["maxclients"]);
      if (globalSection.has("maxqueue"))
	configuration.maxQueue=parseAsUnsignedInt(globalSection["maxqueue"]);
      if (globalSection.has("loopdelay"))
	configuration.loopDelay=parseAsUnsignedInt(globalSection["loopdelay"]);
      if (globalSection.has("maxinputline"))
	configuration.maxLine=parseAsUnsignedInt(globalSection["maxinputline"]);
      if (globalSection.has("datadir"))
	{
	  std::string dataDir=trim(globalSection["datadir"]);
	  if (!dataDir.empty() && dataDir[dataDir.length()-1]!='/')
	    dataDir+='/';
	  configuration.dataDir=dataDir;
	}
      if (globalSection.has("logfilename"))
	configuration.logFileName=trim(globalSection["logfilename"]);
      if (globalSection.has("loglevel"))
	configuration.logLevel=parseAsUnsignedInt(globalSection["loglevel"]);
      if (globalSection.has("logtostderr"))
	configuration.logToStdErr = parseAsBool(globalSection["logtostderr"]);
    }
  int i;
  for (i=0;i<configFile.getSectionCount();i++)
    {
      if (configFile.getSection(i).getName()!="output")
	continue;
      OutputConfiguration outputConfiguration;
      outputConfiguration.init(configFile.getSection(i));
  configuration.outputs.push_back(outputConfiguration);
    }
  for(i=0;i<configuration.outputs.size();i++)
    for(int j=i+1;j<configuration.outputs.size();j++)
      if (configuration.outputs[i].name==configuration.outputs[j].name)
	throw ConfigurationException("there are outputs with the same name \'"+configuration.outputs[i].name+"\'");
  if (configFile.hasSection("default"))
    {
      const ConfigFileSection& defaultSection=configFile.findSection("default");
      if (defaultSection.has("output"))
	configuration.defaultOutputName=defaultSection["output"];
      if (defaultSection.has("chars"))
	{
	  std::wstring chars;
	  if (!decodeUTF8(defaultSection["chars"], chars))
	    throw UTF8Exception("Parameter \'chars\' for section \'default\' in your configuration file contains illegal UTF8 sequence");
	  configuration.defaultOutputChars=chars;
	}
    }
  if (configuration.daemon)
    configuration.logToStdErr=0;
  if (configuration.daemon && !trim(configuration.toSay).empty())
    throw ConfigurationException("You should not use parameters \'--daemon\' and \'--say\' together.");
}

static void initOutput()
{
  for(int i=0;i<configuration.outputs.size();i++)
    {
      const OutputConfiguration& output=configuration.outputs[i];
      if (output.type==OutputConfiguration::COMMAND)
	{
	  auto_ptr<CommandPlayer> commandPlayer(new CommandPlayer(output.name));
	  commandPlayer->init(output);
	  auto_ptr<Player> player(commandPlayer.release());
	  tqAddPlayer(player);
	  logMsg(LOG_INFO, "Added command output \'%s\'", output.name.c_str());
	  continue;
	}
      assert(0);
    }
  tqRun(configuration.maxQueue, configuration.loopDelay);
}

static void closeOutput()
{
}

static void closeSockets()
{
  std::list<Socket*>::iterator it;
  for(it=toListen.begin();it!=toListen.end();it++)
    {
      (*it)->close();
      delete *it;
    }
  toListen.clear();
}

static void initSockets()
{
  if (!configuration.useInetSocket && trim(configuration.unixSocket).empty())
    throw ConfigurationException("no sockets are specified in the configuration file to listen");
  closeSockets();
  if (!trim(configuration.unixSocket).empty())
    {
      auto_ptr<UnixSocket> socket(new UnixSocket());
      socket->open(trim(configuration.unixSocket));
      toListen.push_back(socket.get());
      socket.release();
      logMsg(LOG_INFO, "Unix domain socket was opened as \'%s\'", trim(configuration.unixSocket).c_str());
    }
  if (configuration.useInetSocket)
    {
      auto_ptr<InetSocket> socket(new InetSocket());
      socket->open(configuration.inetPort);
      toListen.push_back(socket.get());
      socket.release();
      logMsg(LOG_INFO, "Accepting TCP/IP connections at port %d", configuration.inetPort);
    }
}

static int run()
{
  logMsg(LOG_INFO, "Charset for I/O operation: %s", transcoding.getIOCharset().c_str());
  int exitCode=0;
  try {
    languages.load(configuration.dataDir);
    initOutput();
    logMsg(LOG_INFO, "All outputs are ready!");
    Dispatcher dispatcher(configuration.prepareDispatcherParams());
    logMsg(LOG_INFO, "Dispatcher was created.");
    dispatcher.initTextProcessor(configuration);
    logMsg(LOG_INFO, "Dispatcher is initialized.");
    initSockets();
    logMsg(LOG_INFO, "All sockets are ready!");
    if (!trim(configuration.pidFileName).empty())
      {
	logMsg(LOG_INFO, "Saving pid to \'%s\'", configuration.pidFileName.c_str());
	std::ofstream f(trim(configuration.pidFileName).c_str());
	if (!f)
	  logMsg(LOG_ERROR, "Could not open \'%s\' for pid saving", trim(configuration.pidFileName).c_str()); else
	    f << getpid();
      }
    if (configuration.daemon)
      logMsg(LOG_INFO, "server is ready to process speech in daemon mode"); else
	logMsg(LOG_INFO, "Server is started in non-daemon mode");
    if (!trim(configuration.startUpMessage).empty())
      {
	Client client;
	dispatcher.text(client, trim(configuration.startUpMessage));
      }
    exitCode=dispatcher.run(toListen);
  }
  catch(Exception& e)
    {
      e.makeLogReport();
      logMsg(LOG_INFO, "Stopping server after exception handling");
      exitCode=2;
    }
  catch(std::bad_alloc)
    {
      logMsg(LOG_FATAL, "No free memory for server operations. Exiting...");
      exitCode=4;
    }
  closeSockets();
  closeOutput();
    return exitCode;
}

static int sayMode(const std::wstring& text)
{
  if (trim(text).empty())
    {
      logMsg(LOG_ERROR, "Server was started in \'say\' mode, but there is no any text to say.");
      return 1;
    }
  int exitCode=0;
  try {
    languages.load(configuration.dataDir);
    initOutput();
    Dispatcher dispatcher(configuration.prepareDispatcherParams());
    dispatcher.initTextProcessor(configuration);
    logMsg(LOG_INFO, "Server is started in \'say\' mode. (text: %s)", WString2IO(text).c_str());
    Client client;
    tqIdle=0;
    dispatcher.text(client, text);
    while(!tqIdle);
  }
  catch(Exception& e)
    {
      e.makeLogReport();
      logMsg(LOG_INFO, "Stopping server after exception handling");
      exitCode=2;
    }
  catch(std::bad_alloc)
    {
      logMsg(LOG_FATAL, "No free memory for server operations. Exiting...");
      exitCode=4;
    }
  closeOutput();
  return exitCode;
}

static void printConfiguration()
{

}

int main(int argc, char *argv[])
{
  try {
    loadConfiguration(argc, argv);
  }
  catch(ConfigurationException& e)
    {
      std::cerr << "Sorry, there are some server configuration problems." << std::endl;
      std::cerr << "You should investigate it, before running voicemand again." << std::endl << std::endl;
      std::cerr << e.getMessage() << std::endl << std::endl;
      std::cerr << "Please, check your configuration and try again." << std::endl;
      return 1;
    }
  catch(UTF8Exception& e)
    {
      std::cerr << "Sorry, there are some problems with UTF-8 processing." << std::endl;
      std::cerr << "Probably, there is some character sequence encoded with other coding system." << std::endl << std::endl;
      std::cerr << e.getMessage() << std::endl << std::endl;
      std::cerr << "Please, fix this problem and try again." << std::endl;
      return 1;
    }
  catch(SystemException& e)
    {
      std::cerr << "Sorry, there was system call fail while reading and parsing your configuration file." << std::endl;
      std::cerr << "You should investigate this problem before running voicemand next tine." << std::endl << std::endl;
      std::cerr << e.getMessage() << std::endl << std::endl;
      std::cerr << "Probably your configuration file is placed in wrong place or you don\'t have permissions to read it." << std::endl;
      return 1;
    }
  catch(ConfigFileException& e)
    {
      std::cerr << "Sorry, there are some problems with configuration file processing." << std::endl;
      std::cerr << "The following error was found:" << std::endl << std::endl;
      std::cerr << e.getMessage() << std::endl << std::endl;
      std::cerr << "Server cannot run with this error. Exiting." << std::endl;
      return 1;
    }
  catch(ConfigFileValueTypeException& e)
    {
      std::cerr  << "Sorry, there are some problems with configuration file processing." << std::endl;
      std::cerr << "Type mismatch in one or more parameters:" << std::endl << std::endl;
      std::cerr << e.getMessage() << std::endl << std::endl;
      std::cerr << "Please, fix this problem and try again." << std::endl;
      return 1;
    }
  configLogConsole=configuration.logToStdErr;
  configLogFileName=configuration.logFileName;
  configLogLevel=configuration.logLevel;
  if (configuration.testConfiguration)
    {
      configuration.printConfiguration();
      return 0;
    }
  if (!trim(configuration.toSay).empty())
    return sayMode(trim(configuration.toSay));
  if (!configuration.daemon)
    {
      printLogo(0);
      return run();
    }
  pid_t p=fork();
  if (p==(pid_t)-1)
    {
      std::cerr << "Sorry, there are some problems with starting the server in daemon mode." << std::endl;
      std::cerr << "Could not create new process:" << strerror(errno) << std::endl;
      return 1;
    }
  if (p == (pid_t)0)
    {
      logMsg(LOG_INFO, "New process started.");
      exit(run());
    }
  return 0;
}
