/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"
#include"sockets.h"
#include"clients.h"
#include"protocol.h"

void VoicemanProtocol::line(const std::wstring& s, Client& client)
{
  wchar_t cmd;
  std::wstring arg;
  if (!split(s, cmd, arg))
    {
      logMsg(LOG_ERROR, "Protocol parse error:invalid line:%s", WString2IO(s).c_str());
      return;
    }
  switch(cmd)
    {
    case 'T':
      text(client, arg);
      break;
    case 'L':
      prLetter(arg, client);
      break;
    case 'S':
      stop(client);
      break;
    case 'V':
      prParam(PARAM_VOLUME, arg, client);
      break;
    case 'P':
      prParam(PARAM_PITCH, arg, client);
      break;
    case 'R':
      prParam(PARAM_RATE, arg, client);
      break;
    case 'B':
      prTone(arg, client);
      break;
    default:
      logMsg(LOG_ERROR, "Received unknown command. (\'%d\')", cmd);
    } //swithc;
}

bool VoicemanProtocol::split(const std::wstring& s, wchar_t& cmd, std::wstring& arg)
{
  if (s.length() < 2)
    return 0;
  if (s[1] != ':')
    return 0;
  cmd=s[0];
  arg.erase();
  for(int i=2;i<s.length();i++)
    arg+=s[i];
  return 1;
}

void VoicemanProtocol::prLetter(const std::wstring& value, const Client& client)
{
  if (value.length() != 1)
    {
      logMsg(LOG_ERROR, "Argument of LETTER command has an invalid length. (arg=%s)", WString2IO(value).c_str());
      return;
    }
  letter(client, value[0]);
}

void VoicemanProtocol::prParam(int paramType, const std::wstring& value, Client& client)
{
  if (!checkTypeUnsignedInt(value))
    {
      logMsg(LOG_ERROR, "Received illegal parameter value. \'%s\' is not an unsigned integer number.", WString2IO(value).c_str());
      return;
    }
  TextParam p(parseAsUnsignedInt(value));
  param(client, paramType, p);
}

void VoicemanProtocol::prTone(const std::wstring& value, const Client& client)
{
  std::wstring p1, p2;
  p1=trim(getDelimitedSubStr(value, 0, ':'));
  p2=trim(getDelimitedSubStr(value, 1, ':'));
  if (p1.empty() || p2.empty())
    {
      logMsg(LOG_ERROR, "Missed one of the parameters for the \'TONE\' command.");
      return;
    }
  if (!checkTypeUnsignedInt(p1) || !checkTypeUnsignedInt(p2))
    {
      logMsg(LOG_ERROR, "\'TONE\' command has invalid parameters.");
      return;
    }
  tone(client, parseAsUnsignedInt(p1), parseAsUnsignedInt(p2));
}

