/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"
#include"config.h"
#include"configuration.h"
#include"player.h"
#include"command.h"
#include"queue.h"
#include"tone.h"

static int tqLimit, tqDelay;
volatile bool tqIdle=1;

enum {TEXT_ITEM = 0, TONE_ITEM = 1};

struct QUEUEITEM
{
  QUEUEITEM(const TextItem& item): type(TEXT_ITEM), textItem(item) {}
  QUEUEITEM(int fr, int len): type(TONE_ITEM), freq(fr), lengthMs(len) {}

  int type;
  TextItem textItem;
  int freq, lengthMs;
}; //struct QUEUEITEM;

static std::list<QUEUEITEM> tq;
static pthread_mutex_t tqMutex;
static pthread_t tqThread;
static std::vector<Player*> tqPlayers;

static TonePlayer tonePlayer;

static void *tqProcess(void *param)
{
  while(1)
    {
      usleep(tqDelay);
      pthread_mutex_lock(&tqMutex);
      if (tq.size()==0)
	{
	  if (!tqIdle)
	    {
	      int i;
	      for(i=0;i<tqPlayers.size();i++)
		if (tqPlayers[i]->processing())
		  break;
	      if (i>=tqPlayers.size())
		tqIdle=1;
	    } // idle;
	  pthread_mutex_unlock(&tqMutex);
	  continue;
	}//no queue items;
      int i;
      for(i=0;i<tqPlayers.size();i++)
	if (tqPlayers[i]->processing())
	  break;
      if (i<tqPlayers.size())
	{
	  pthread_mutex_unlock(&tqMutex);
	  continue;
	}
      QUEUEITEM item=tq.front();
      tq.pop_front();
      switch(item.type)
	{
	case TEXT_ITEM:
	  for(i=0;i<tqPlayers.size();i++)
	    {
	      if (tqPlayers[i]->getName()==item.textItem.getOutputName())
		{
		  tqPlayers[i]->play(item.textItem);
		  break;
		} // if name;
	    } // for players;
	  break;
	case TONE_ITEM:
	  tonePlayer.play(item.freq, item.lengthMs);
	  break;
	default:
	  assert(0);
	} // switch;
      pthread_mutex_unlock(&tqMutex);
    } // endless loop;
}

void tqAddPlayer(auto_ptr<Player> player)
{
  assert(player.get()!=NULL);
  assert(!trim(player->getName()).empty());
  tqPlayers.push_back(player.get());
  player.release();
}

void tqRun(int limit, int delay)
{
  assert(delay>=10);
  assert(limit >0);
  tqLimit=limit;
  tqDelay=delay;
  VM_SYS(pthread_mutex_init(&tqMutex, NULL) != -1, "pthread_mutex_init()");
  VM_SYS(pthread_create(&tqThread, NULL, tqProcess, NULL) != -1, "pthread_create()");
}

void tqAddText(const TextItem& textItem)
{
  pthread_mutex_lock(&tqMutex);
  if (tqLimit > 0 && tq.size() >= tqLimit)
    {
      logMsg(LOG_WARN, "Ignoring text item, because reached queue length limit");
      pthread_mutex_unlock(&tqMutex);
      return;
    }
  tq.push_back(QUEUEITEM(textItem));
  pthread_mutex_unlock(&tqMutex);
}

void tqAddTone(int freq, int lengthMs)
{
  assert(freq >= 10);
  assert(lengthMs >= 0);
  pthread_mutex_lock(&tqMutex);
  if (tqLimit > 0 && tq.size() >= tqLimit)
    {
      logMsg(LOG_WARN, "Ignoring tone in queue, because reached queue length limit");
      pthread_mutex_unlock(&tqMutex);
      return;
    }
  tq.push_back(QUEUEITEM(freq, lengthMs));
  pthread_mutex_unlock(&tqMutex);
}

void tqStop()
{
  pthread_mutex_lock(&tqMutex);
  tq.clear();
  int i;
  for(i=0;i<tqPlayers.size();i++)
    tqPlayers[i]->stop();
  pthread_mutex_unlock(&tqMutex);
}
