/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"
#include"rus.h"

static void addVectorItem(const std::string& str, std::vector<std::wstring>& items)
{
  std::string s=getDelimitedSubStr(str, 1, ':');
  if (trim(s).empty())
    throw ConfigurationException("There is line with empty second item in file with Russian constants");
  if (!checkTypeUnsignedInt(s))
    throw ConfigurationException("There is line with an invalid second item in file with Russian constants. \'"+trim(s)+"\' is not a valid unsigned integer number.");
  int index=parseAsUnsignedInt(s);
  if (items.size()<=index)
    items.resize(index+1);
  if (!decodeUTF8(getDelimitedSubStr(str, 2, ':'), items[index]))
    throw UTF8Exception("There is line in file with Russian constants with an illegal UTF-8 sequence. \'"+getDelimitedSubStr(str, 2, ':')+"\' is not a valid string.");
}

void RusLang::load(const std::string& fileName)
{
  DelimitedFile f;
  f.read(fileName);
  for(int i=0;i<f.getLineCount();i++)
    {
      std::string t=trim(::toLower(f.getItem(i, 0)));
      if (t.empty())
	throw ConfigurationException("There is line with empty first item in file with Russian constants.");
      if (t=="zero")
	{
	  if (!decodeUTF8(f.getItem(i, 1), 	m_zero))
	    throw UTF8Exception("Value for \'zero\' in file with Russian constants contains an illegal UTF-8 sequence.");
	} else
      if (t=="chars")
	{
	  if (!decodeUTF8(f.getItem(i, 1), m_chars))
	    throw UTF8Exception("Value for \'chars\' in file with Russian constants contains an illegal UTF-8 sequence.");
	} else
      if (t=="mlrds")
	addVectorItem(f.getRawLine(i), m_mlrds); else
      if (t=="mlns")
	addVectorItem(f.getRawLine(i), m_mlns); else
      if (t=="thnds")
	addVectorItem(f.getRawLine(i), m_thnds); else
      if (t=="hundreds")
	addVectorItem(f.getRawLine(i), m_hundreds); else
      if (t=="decimals")
	addVectorItem(f.getRawLine(i), m_decimals); else
      if (t=="tens")
	addVectorItem(f.getRawLine(i), m_tens); else
      if (t=="ones")
	addVectorItem(f.getRawLine(i), m_ones); else
      if (t=="onesf")
	addVectorItem(f.getRawLine(i), m_onesF); else
      if (t=="map")
	{
	  std::wstring v1, v2;
	  if (!decodeUTF8(f.getItem(i, 1), v1))
	    throw UTF8Exception("File with Russian constants contains an illegal UTF-8 sequence. \'"+f.getItem(i, 1)+"\' could not be decoded.");
	  if (!decodeUTF8(f.getItem(i, 2), v2))
	    throw UTF8Exception("File with Russian constants contains an illegal UTF-8 sequence. \'"+f.getItem(i, 2)+"\' could not be decoded.");
	  v1=trim(v1);
	  v2=trim(v2);
	  if (v1.length()!=1 || v2.length()!=1)
	    throw ConfigurationException("File with Russian constants contains map item with bad strings. Length of string in such item should be one character. Line: \'"+f.getRawLine(i)+"\'");
	  m_toUpper[v2[0]]=v1[0];
	  m_toLower[v1[0]]=v2[0];
	} else
	throw ConfigurationException("There is line with unknown first item \'"+t+"\' in the file with Russian constants.");
    }
}

int RusLang::getCharType(wchar_t c) const
{
  std::map<wchar_t, wchar_t>::const_iterator it;
  it=m_toUpper.find(c);
  if (it!=m_toUpper.end())
    return LOWCASE;
  it=m_toLower.find(c);
  if (it!=m_toLower.end())
    return UPCASE;
  return OTHER;
}

std::wstring RusLang::getAllChars() const
{
  return m_chars;
}

bool RusLang::equalChars(wchar_t c1, wchar_t c2) const
{
  return toLower(c1)==toLower(c2);
}

wchar_t RusLang::toUpper(wchar_t ch) const
{
  std::map<wchar_t, wchar_t>::const_iterator it=m_toUpper.find(ch);
  if (it==m_toUpper.end())
    return ch;
  return it->second;
}

wchar_t RusLang:: toLower(wchar_t ch) const
{
  std::map<wchar_t, wchar_t>::const_iterator it=m_toLower.find(ch);
  if (it==m_toLower.end())
    return ch;
  return it->second;
}

std::wstring RusLang::toUpper(const std::wstring& str) const
{
  std::wstring s;
  for(int i=0;i<str.length();i++)
    s+=toUpper(str[i]);
  return s;
}

std::wstring RusLang::toLower(const std::wstring& str) const
{
  std::wstring s;
  for(int i=0;i<str.length();i++)
    s+=toLower(str[i]);
  return s;
}

std::wstring RusLang::processHundred(const std::wstring& inStr, const std::vector<std::wstring>& items, bool female) const
{
  std::wstring s, str = inStr;
  int i;
  assert(str.length()<=3);
  while(str.length() < 3)
    str = L'0' + str;
  for(i=0;i<3;i++)
    if (str[i] != '0')
      break;
  if (i==str.length())
    return std::wstring();
  if (str[0] != '0')
    attachString(s, m_hundreds[str[0]-'0']);
  if (str[1] != '0' && str[1] != '1')
    attachString(s, m_decimals[str[1]-'0']);
  if (str[1] == '1')
    {
      attachString(s, m_tens[str[2]-'0']);
    } else
    {
      if (str[2] != '0')
	{
	  if (female)
	    attachString(s, m_onesF[str[2]-'0']); else
	    attachString(s, m_ones[str[2]-'0']);
	}
    }
  if (items.empty())
    return s;
  if (str[1] == '1')
    attachString(s, items[2]); else
    {
      if (str[2] == '1')
	attachString(s, items[0]); else
	if (str[2] >= '2' && str[2] <= '4')
	  attachString(s, items[1]); else
	  attachString(s, items[2]);
    }
  return s;
}

std::wstring RusLang::digitsToWords(const std::wstring& inStr) const
{
  int i;
  std::vector<std::wstring> sList;
  std::wstring str;
  assert(!inStr.empty());
  for(i=0;i<inStr.length();i++)
    {
      assert(inStr[i] >= '0' && inStr[i] <= '9');
      if (inStr[i] != '0')
	break;
    }
  if (i==inStr.length())
    return m_zero;
  bool accepting = 0;
  for(i=0;i<inStr.length();i++)
    {
      if (inStr[i] != '0')
	accepting = 1;
      if (accepting)
	str += inStr[i];
    }
  while(str.length())
    {
      if (str.length() >= 3)
	{
	  std::wstring ss;
	  ss += str[str.length()-3];
	  ss += str[str.length()-2];
	  ss += str[str.length()-1];
	  sList.push_back(ss);
	  str.resize(str.size()-3);
	  continue;
	}
      if (str.length() == 2)
	{
	  std::wstring ss;
	  ss += str[0];
	  ss += str[1];
	  sList.push_back(ss);
	  str.erase();
	  continue;
	}
      if (str.length() == 1)
	{
	  std::wstring ss;
	  ss = str[0];
	  sList.push_back(ss);
	  str.erase();
	  continue;
	}
    }
  int j;
  str.erase();
  for(j=sList.size()-1;j>=0;j--)
    {
      if (j>3)
	attachString(str, processHundred(sList[j], std::vector<std::wstring>(), 0) ); else
	if (j == 3)
	  attachString(str, processHundred(sList[j], m_mlrds, 0) ); else
	  if (j == 2)
	    attachString(str, processHundred(sList[j], m_mlns, 0)); else
	    if (j == 1)
	      attachString(str, processHundred(sList[j], m_thnds, 1)); else
	      attachString(str, processHundred(sList[j], std::vector<std::wstring>(), 0));
    }
  return str;
}

void RusLang::expandNumbers(std::wstring& str, bool singleDigits) const
{
  std::wstring inStr=str;
  str.erase();
  if (singleDigits)
    {
      int i;
      bool b = 0;
      for(i=0;i<inStr.length();i++)
	{
	  if (inStr[i] >= '0' && inStr[i] <= '9')
	    {
	      b = 1;
	      if (inStr[i] == '0')
		attachString(str, m_zero); else
		attachString(str, m_ones[inStr[i]-'0']);
	    } else
	    {
	      if (b)
		{
		  str += ' ';
		  b = 0;
		}
	      str += inStr[i];
	    }
	}
      return;
    }
  int i;
  bool d = 0;
  std::wstring sStr;
  for(i=0;i<inStr.length();i++)
    {
      if (inStr[i]>= '0' && inStr[i] <= '9')
	{
	  d = 1;
	  sStr += inStr[i];
	} else
	{
	  if (d)
	    {
	      attachString(str, digitsToWords(sStr));
	      d = 0;
	      sStr.erase();
	      str += ' ';
	    }
	  str += inStr[i];
	}
    }
  if (d)
    attachString(str, digitsToWords(sStr));
}

std::wstring RusLang::separate(const std::wstring& text) const
{
  return text;
}

void RusLang::markCapitals(const std::wstring& text, std::vector<bool>& marks) const
{

}
