/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"
#include"sockets.h"

void Socket::close()
{
  if (!m_opened)
    return;
  ::close(m_sock);
  m_opened=0;
};

int Socket::read(void* buf, int s) const
{
  assert(m_opened);
  int r = ::read(m_sock, buf, s);
  VM_SYS(r!=-1, "read()");
  return r;
}

int Socket::write(const void *buf, int s) const
{
  assert(m_opened);
  int r = ::write(m_sock, buf, s);
  VM_SYS(r!=-1, "write()");
  return r;
}

bool Socket::read(std::string &s) const
{
  char buf[2048];
  int c=read(buf, sizeof(buf));
  if (c==0)
    return 0;
  int i;
  s.erase();
  for(i=0;i<c;i++)
    s+=buf[i];
  return 1;
}

void Socket::write(const std::string &s) const
{
  int k = s.length();
  do {
    char buf[2048];
    int sentCount;
    int i;
    if (k >= sizeof(buf))
      {
	for(i=0;i<sizeof(buf);i++)
	  buf[i] = s[s.length()-k+i];
	sentCount = write(buf, sizeof(buf));
	k-=sentCount;
      } else
	{
	  for(i=0;i<k;i++)
	    buf[i]=s[s.length()-k+i];
	  sentCount = write(buf, k);
	  k -= sentCount;
	}
  } while(k);
}

void UnixSocket::open(const std::string &name)
{
  assert(!m_opened);
  unlink(name.c_str());
  struct sockaddr_un addr;
  int fd = socket(AF_UNIX, SOCK_STREAM, 0);
  VM_SYS(fd!=-1, "socket(AF_UNIX, SOCK_STREAM, 0)");
  addr.sun_family=AF_UNIX;
  strncpy(addr.sun_path, name.c_str(), sizeof(addr.sun_path));
  if (bind(fd, (struct sockaddr*)&addr, sizeof(sockaddr_un)) == -1)
    {
      ::close(fd);
      throw SystemException("bind()");
    }
  if (listen(fd, 512) == -1)
    {
      ::close(fd);
      throw SystemException("listen()");
    }
  chmod(name.c_str(), 0666);
  m_sock=fd;
  m_opened=1;
}

void InetSocket::open(int port)
{
  assert(!m_opened);
  logMsg(LOG_TRACE, "Opening inet socket at port %d", port);
  int fd=socket(AF_INET, SOCK_STREAM, 0);
  VM_SYS(fd!=-1, "socket(AF_INET, SOCK_STREAM, 0)");
  logMsg(LOG_TRACE, "socket() call completed");
  struct sockaddr_in addr;
  bzero(&addr, sizeof(struct sockaddr_in));
  addr.sin_family=AF_INET;
  addr.sin_addr.s_addr=htonl(INADDR_ANY);
  addr.sin_port=htons(port);
  if (bind(fd, (struct sockaddr*)&addr, sizeof(struct sockaddr_in)) == -1)
    {
      ::close(fd);
      logMsg(LOG_TRACE, "socket was closed after bind() call failure (%s)", strerror(errno));
      throw SystemException("bind()");
    }
  logMsg(LOG_TRACE, "bind() call completed");
  if (listen(fd, 512) == -1)
    {
      ::close(fd);
      throw SystemException("listen()");
    }
  m_sock=fd;
  m_opened=1;
}
