/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#ifndef __VOICEMAN_STRING_H__
#define __VOICEMAN_STRING_H__

#define BLANK_CHAR(x) ((x)==10 || (x)==13 || (x)==9 || (x)==32)
#define DIGIT_CHAR(x) ((x)>='0' && (x)<='9')

template<class T>
T trim(const T& str)
{
  int l1=0, l2=str.length();
  while(l1<str.length() && BLANK_CHAR(str[l1]))
    l1++;
  while(l2>l1 && BLANK_CHAR(str[l2-1]))
    l2--;
  T newStr;
  for(int i=l1;i<l2;i++)
    newStr+=str[i];
  return newStr;
}

template<class T>
T toUpper(const T& str)
{
  int i;
  T s=str;
  for(i=0;i<s.length();i++)
    if (s[i]>='a' && s[i]<='z')
      s[i] = 'A' + (s[i] - 'a');
  return s;
}

template<class T>
T toLower(const T& str)
{
  int i;
  T s=str;
  for(i=0;i<s.length();i++)
    if (s[i]>='A' && s[i]<='Z')
      s[i]= 'a'+(s[i] - 'A');
  return s;
}

template<class T>
bool contains(const T& str, const T& s)
{
  int i;
  for(i=0;i<str.length();i++)
    {
      int j;
      for(j=0;j<s.length();j++)
	if (str[i]==s[j])
	  return 1;
    }
  return 0;
};

template<class T>
T adjust(const char* str)
{
  assert(str!=NULL);
  T s;
  for(int i=0;str[i]!='\0';i++)
    s+=str[i];
  return s;
}

template<class T>
bool checkTypeBool(const T& s)
{
  T ss=trim(toLower(s));
  if (ss==adjust<T>("yes") ||
      ss==adjust<T>("no") ||
      ss==adjust<T>("true") ||
      ss==adjust<T>("false") ||
      ss==adjust<T>("0") ||
      ss==adjust<T>("1"))
    return 1;
  return 0;
}

template<class T>
bool checkTypeUnsignedInt(const T& s)
{
  T ss=trim(s);
  if (ss.empty())
    return 0;
  int i=0;
  if (ss[0]=='+')
    i=1;
  if (i>=ss.length())
    return 0;
  for(;i<ss.length();i++)
    if (ss[i]<'0' || ss[i]>'9')
      return 0;
  return 1;
}

template<class T>
bool checkTypeInt(const T& s)
{
  T ss=trim(s);
  if (ss.empty())
    return 0;
  int i=0;
  if (ss[0]=='+' || ss[0]=='-')
    i=1;
  if (i>=ss.length())
    return 0;
  for(;i<ss.length();i++)
    if (ss[i]<'0' || ss[i]>'9')
      return 0;
  return 1;
}

template<class T>
bool parseAsBool(const T& s)
{
  T ss=trim(toLower(s));
  if (ss==adjust<T>("true") || ss==adjust<T>("yes") || ss==adjust<T>("1"))
    return 1;
  if (ss==adjust<T>("false") || ss==adjust<T>("no") || ss== adjust<T>("0"))
    return 0;
  assert(0);
}

template<class T>
unsigned int parseAsUnsignedInt(const T& s)
{
  T ss=trim(s);
  assert(!ss.empty());
  int n=0;
  int i=0;
  if (ss[0]=='+')
    i=1;
  assert(i<ss.length());
  for(;i<ss.length();i++)
    {
      assert(ss[i]>='0' || ss[i] <='9');
      n *= 10;
      n += ss[i]-'0';
    }
  return n;
}

template<class T>
int parseAsInt(const T& s)
{
  T ss=trim(s);
  assert(!ss.empty());
  int n=0;
  int i=0;
  bool b=0;
  if (ss[0]=='-')
    {
      b=1;
      i=1;
    } else
      if (ss[0]=='+')
	i=1;
  assert(i<ss.length());
  for(;i<ss.length();i++)
    {
      assert(ss[i]>='0' && ss[i] <='9');
      n *= 10;
      n += s[i]-'0';
    }
  if (b)
    n *= -1;
  return n;
}

template<class T>
bool checkTypeDouble(const T& s)
{
  T ss=trim(s);
  int state=0;
  for(int i=0;i<ss.length();i++)
    {
      if (state==0 &&(ss[i]=='-' || ss[i]=='+' || (ss[i]>='0' && ss[i]<='9')))
	{
	  state=1;
	  continue;
	}
      if (state==1 && ss[i]>='0' && ss[i]<='9')
	continue;
      if (state==1 && ss[i]=='.')
	{
	  state=2;
	  continue;
	}
      if (state==2 && ss[i]>='0' && ss[i]<='9')
	{
	  state=3;
	  continue;
	}
      if (state==3 && ss[i]>='0' && ss[i]<='9')
	continue;
      return 0;
    }// for;
  return (state==1 || state==3);
}

template<class T>
double parseAsDouble(const T& s)
{
  assert(checkTypeDouble(s));
  T s1, s2, ss=trim(s);
  bool b=0, sign=0;
  int i;
  for(i=0;i<ss.length();i++)
    {
      if (ss[i]=='.')
	{
	  b=1;
	  continue;
	}
      if (ss[i]=='-')
	{
	  sign=1;
	  continue;
	}
      if (ss[i]<'0' || ss[i]>'9')
	continue;
      if (!b)
	s1+=ss[i];else 
	  s2+=ss[i];
    }
  assert(!s1.empty());
  double d1=0, d2=0;
  for(i=0;i<s1.length();i++)
    {
      d1*=10;
      d1+=s1[i]-'0';
    }
  if (!s2.empty())
    {
      int z=s2.length()-1;
      while(z>=0 && s2[z]=='0') z--;
      if (z>=0)
	{
	  for(i=0;i<=z;i++)
	    {
	      d2*=10;
	      d2+=s2[i]-'0';
	    }
	  for(i=0;i<=z;i++)
	    d2/=10;
	}
    }
  d1+=d2;
  if (sign)
    d1*= -1;
  return d1;
}

template<class T>
T makeStringFromDouble(double value, int format)
{
  assert(format >= 0 && format<=10);
  double ff=value;
  int i=format;
  for(;i>0;i--)
    ff*=10;
  long long c=(long long)ff;
  bool sign=c<0;
  if (sign)
    c*=-1;
  std::ostringstream ss;
  ss<<c;
  T s=adjust<T>(ss.str().c_str());
  while(s.length() < format+1)
    s = '0'+s;
  if (sign)
    s='-'+s;
  if (format==0)
    {
      return s;
    }
  s+=' ';
  for(i=s.length()-1;i>=s.length()-format;i--)
    s[i]=s[i-1];
  s[s.length()-format-1]='.';
  return s;
}

template<class T>
void attachSpace(T& str)
{
  if (str.empty())
    return;
  if (BLANK_CHAR(str[str.length()-1]))
    return;
  str+=' ';
}

template<class T>
void attachString(T& str, const T& toAttach)
{
  if (trim(str).empty())
    {
      str=trim(toAttach);
      return;
    }
  if (BLANK_CHAR(str[str.length()-1]))
    {
      str+=trim(toAttach);
      return;
    }
  str+=' ';
  str+=trim(toAttach);
}

template<class T>
void attachCharWithoutDoubleSpaces(T& str, typename T::value_type ch)
{
  if (!BLANK_CHAR(ch))
    {
      str+=ch;
      return;
    }
  if (BLANK_CHAR(str[str.length()-1]))
    return;
  str+=' ';
}

template<class T>
void attachStringWithSpace(T& str, const T& toAttach)
{
  str+=trim(toAttach);
  str+=' ';
}

template<class T>
void attachCharWithSpace(T& str, typename T::value_type ch)
{
  if (!BLANK_CHAR(ch))
    str+=ch;
  str+=' ';
}

template<typename T>
T getDelimitedSubStr(const T& s, int index, char delimiter)
{
  assert(index >= 0);
  T ss=trim(s);
  int i, k;
  for(i=0,k=0;i<ss.length() && k<index;i++)
    if (ss[i] == delimiter)
      k++;
  if (i>=ss.length())
    return T();
  T s1;
  for(;i<ss.length() && ss[i]!= delimiter;i++)
    s1 += ss[i];
  return s1;
}

namespace __gnu_cxx
{
  template<> struct hash<wchar_t>
  {
    size_t operator()(wchar_t x) const
    {
      return hash< size_t>()(x);
    }
  };
} //namespace __gnu_cxx;

template<class T>
class StringIterator 
{
public:
  typedef typename T::value_type CHAR;

  StringIterator(const T& str, const T& chars): m_start(-1), m_str(str), m_end(-1)
  {
    assert(!chars.empty());
    for(int i=0;i<chars.length();i++)
      m_chars.insert(chars[i]);
  }

    bool match(CHAR c) const
  {
    typename hash_set<CHAR>::const_iterator i=m_chars.find(c);
    return i!=m_chars.end();
  }

  bool next()
  {
    m_start=m_end+1;
    if (m_start>=m_str.length())
      return 0;
    while(m_start<m_str.length() && !match(m_str[m_start])) m_start++;
    if (m_start>=m_str.length())
      return 0;
    m_end=m_start;
    while(m_end<m_str.length() && match(m_str[m_end])) m_end++;
    return 1;
  }

  T str() const
  {
    assert(m_start>=0 && m_end <= m_str.length());
    T s;
    for(int i=m_start;i<m_end;i++)
      s+=m_str[i];
    return s;
  }

  int start() const 
  { 
    assert(m_start>=0 && m_end<=m_str.length()); 
    return m_start; 
  }

  int end() const 
  { 
    assert(m_start>=0 && m_end <=m_str.length()); 
    return m_end; 
  }

private:
  const T& m_str;
  hash_set<CHAR> m_chars;
  int m_start, m_end;
};//class StringIterator;

template<class T>
class StringDelimitedIterator 
{
public:
  typedef typename T::value_type CHAR;

  StringDelimitedIterator(const T& str, const T& chars): m_start(-1), m_str(str), m_end(-1)
  {
    assert(!chars.empty());
    for(int i=0;i<chars.length();i++)
      m_chars.insert(chars[i]);
  }

    bool match(CHAR c) const
  {
    typename hash_set<CHAR>::const_iterator i=m_chars.find(c);
    return i==m_chars.end();
  }

  bool next()
  {
    m_start=m_end+1;
    if (m_start>=m_str.length())
      return 0;
    while(m_start<m_str.length() && !match(m_str[m_start])) m_start++;
    if (m_start>=m_str.length())
      return 0;
    m_end=m_start;
    while(m_end<m_str.length() && match(m_str[m_end])) m_end++;
    return 1;
  }

  T str() const
  {
    assert(m_start>=0 && m_end <= m_str.length());
    T s;
    for(int i=m_start;i<m_end;i++)
      s+=m_str[i];
    return s;
  }

  int start() const 
  { 
    assert(m_start>=0 && m_end<=m_str.length()); 
    return m_start; 
  }

  int end() const 
  { 
    assert(m_start>=0 && m_end <=m_str.length()); 
    return m_end; 
  }

private:
  const T& m_str;
  hash_set<CHAR> m_chars;
  int m_start, m_end;
};//class StringDelimitedIterator;

#endif //__VOICEMAN_STRING_H__
