/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#include"voiceman.h"

#define CAP_OVERHEAD 20

struct REPLACEMENT 
{
  REPLACEMENT(int output, const std::wstring& oldString, const std::wstring& newString): outputIndex(output),oldValue(oldString), newValue(newString) {}

  int outputIndex;
  std::wstring oldValue, newValue;
}; //struct REPLACEMENT;

class Processor: public TextProcessor
{
public:
  Processor(int digitsMode, bool capitalization, bool separation): m_digitsMode(digitsMode), m_capitalization(capitalization), m_separation(separation) {}
  virtual ~Processor() {}

  void setDefaultOutput(const std::string& outputName)
  {
    assert(!trim(outputName).empty());
    m_defaultOutput=outputName;
  }

  void setLang(const std::string& outputName, const Lang* lang)
  {
    int outputIndex=ensureOutput(outputName);
    assert(m_outputs.size()==m_langs.size());
    m_langs[outputIndex]=lang;
  }

  void associate(const std::wstring& str, const std::string& outputName)
  {
    int outputIndex;
    if (trim(outputName)==DEFAULT_OUTPUT_NAME)
      outputIndex = -1; else
	outputIndex=ensureOutput(outputName);
    for(int i=0;i<str.length();i++)
      m_charsTable[str[i]]=outputIndex;
  }

  void addReplacement(const std::string& outputName, const std::wstring& oldValue, const std::wstring& newValue)
  {
    int outputIndex=ensureOutput(outputName);
    m_replacements.push_back(REPLACEMENT(outputIndex, oldValue, newValue));
  }

  void setSpecialValueFor(wchar_t c, const std::wstring& value)
  {
    m_specialValues[c]=trim(value);
  }

  void process(const TextItem& text, std::list<TextItem>& items) const;
  void processLetter(wchar_t c, TextParam volume, TextParam pitch, TextParam rate, std::list<TextItem>& items);

private:
  void split(const std::wstring& text, std::list<TextItem>& items) const;
  void processItem(TextItem& text) const;
  bool findReplacement(const std::wstring& str, int pos, int outputIndex, int& result) const;
  std::wstring insertReplacements(const std::wstring& str, int outputIndex) const;

  int getDefaultOutputIndex() const
  {
    if (trim(m_defaultOutput).empty())
      return -1;
    for(int i=0;i<m_outputs.size();i++)
      if (m_outputs[i]==m_defaultOutput)
	return i;
    return -1;
  }

  int ensureOutput(const std::string& outputName)
  {
    assert(!trim(outputName).empty());
    for(int i=0;i<m_outputs.size();i++)
      {
	if (m_outputs[i]==trim(outputName))
	  return i;
      }
    m_outputs.push_back(trim(outputName));
    m_langs.push_back(NULL);
    return m_outputs.size()-1;
  }

private:
  std::map<wchar_t, int> m_charsTable;
  std::map<wchar_t, std::wstring> m_specialValues;
  std::vector<REPLACEMENT> m_replacements;
  std::vector<const Lang*> m_langs;
  std::vector<std::string> m_outputs;
  std::string m_defaultOutput;
  int m_digitsMode;
  bool m_separation, m_capitalization;
}; //class Processor;

void Processor::split(const std::wstring& text, std::list<TextItem>& items) const
{
  items.clear();
  int currentOutputIndex=-1, defaultOutputIndex=-1;
  std::wstring currentText;
  for(int i=0;i<text.length();i++)
    {
      wchar_t let = text[i];
      if (BLANK_CHAR(let))
	{
	  if (currentOutputIndex < 0)
	    continue;
	  attachSpace(currentText);
	  continue;
	} // space;
      std::map<wchar_t, int>::const_iterator it=m_charsTable.find(let);
      int outputIndex;
      if (it!=m_charsTable.end())
	{
	  outputIndex = it->second;
	  if (outputIndex<0)
	    outputIndex=defaultOutputIndex;
	  if (outputIndex < 0)
	    outputIndex = getDefaultOutputIndex();;
	} else
	outputIndex=-1;
      if (outputIndex < 0)
	{
	  attachSpace(currentText);
	  continue;
	}
      if (currentOutputIndex >= 0 && currentOutputIndex!=outputIndex)
	{
	  items.push_back(TextItem(m_outputs[currentOutputIndex], currentText));
	  currentText.erase();
	  currentOutputIndex = -1;
	}
      defaultOutputIndex=outputIndex;
      currentOutputIndex=outputIndex;
      currentText+=let;
    } //for;
  if (currentOutputIndex>=0)
    items.push_back(TextItem(m_outputs[currentOutputIndex], currentText));
}

bool Processor::findReplacement(const std::wstring& str, int pos, int outputIndex, int& result) const
{
  assert(pos>=0);
  assert(outputIndex>=0 && outputIndex <= m_outputs.size());
  assert(m_outputs.size()==m_langs.size());
  const Lang* lang=m_langs[outputIndex];
  if (!lang)
    return 0;
  for(int i=0;i<m_replacements.size();i++)
    {
      if (m_replacements[i].outputIndex != outputIndex)
	continue;
      if (str.length() - pos < m_replacements[i].oldValue.length())
	continue;
      int j;
      for(j=0;j<m_replacements[i].oldValue.length();j++)
	if (!lang->equalChars(str[pos+j], m_replacements[i].oldValue[j]))
	    break;
      if (j==m_replacements[i].oldValue.length())
	{
	  result = i;
	  return 1;
	}
    }
  return 0;
}

std::wstring Processor::insertReplacements(const std::wstring& str, int outputIndex) const
{
  assert(outputIndex>=0 && outputIndex<m_outputs.size());
  std::wstring newStr;
  for(int i=0;i<str.length();i++)
    {
      int k;
      if (findReplacement(str, i, outputIndex, k))
	{
	  attachSpace(newStr);
	  newStr += m_replacements[k].newValue;
	  attachSpace(newStr);
	  i += m_replacements[k].oldValue.length()-1;
	} else
	attachCharWithoutDoubleSpaces(newStr, str[i]);
    }
  return newStr;
}

void Processor::processLetter(wchar_t c, TextParam volume, TextParam pitch, TextParam rate, std::list<TextItem>& items)
{
  std::map<wchar_t, int>::const_iterator it=m_charsTable.find(c);
  int outputIndex=-1;
  if (it!=m_charsTable.end())
    {
      outputIndex=it->second;
      if (outputIndex < 0)
	outputIndex=getDefaultOutputIndex();
    }
  if (outputIndex < 0)
    {
      std::map<wchar_t, std::wstring>::const_iterator specialValueIt=m_specialValues.find(c);
      if (specialValueIt!=m_specialValues.end())
	process(TextItem(specialValueIt->second, volume, pitch, rate), items);
      return;
    }
  TextParam p=pitch;
  assert(m_outputs.size()==m_langs.size());
  const Lang* l=m_langs[outputIndex];
  if (l!=NULL && l->getCharType(c)==Lang::UPCASE)
    p+=CAP_OVERHEAD;
  std::map<wchar_t, std::wstring>::const_iterator specialValueIt=m_specialValues.find(c);
  if (specialValueIt!=m_specialValues.end())
    {
      process(TextItem(specialValueIt->second, volume, p, rate), items);
      return;
    }
  std::wstring s;
  s+=c;
  TextItem item(m_outputs[outputIndex], s, volume, p, rate);
  item.mark(0);
  item.setLang(l);
  items.clear();
  items.push_back(item);
}

void Processor::process(const TextItem& text, std::list<TextItem>& items) const
{
  logMsg(LOG_TRACE, "Splitting text \'%s\'", WString2IO(text.getText()).c_str());
  split(text.getText(), items);
  std::list<TextItem>::iterator it;
  for(it=items.begin();it!=items.end();it++)
    {
      it->setVolume(text.getVolume());
      it->setPitch(text.getPitch());
      it->setRate(text.getRate());
      processItem(*it);
    }
}

void Processor::processItem(TextItem& text) const
{
  logMsg(LOG_TRACE, "Processing text item \'%s\'", WString2IO(text.getText()).c_str());
  std::vector<bool> marks;
  int outputIndex;
  for(outputIndex=0;outputIndex<m_outputs.size();outputIndex++)
    if (m_outputs[outputIndex]==text.getOutputName())
      break;
  assert(outputIndex<m_outputs.size());
  std::wstring toSend=insertReplacements(text.getText(), outputIndex);
  assert(m_outputs.size()==m_langs.size());
  const Lang* lang=m_langs[outputIndex];
  if (lang != NULL)
    {
      if (m_separation)
	toSend=lang->separate(toSend);
      switch(m_digitsMode)
	{
	case DIGITS_NORMAL:
	  lang->expandNumbers(toSend, 0);
	  break;
	case DIGITS_SINGLE:
	  lang->expandNumbers(toSend, 1);
	  break;
	}
      marks.resize(toSend.length());
      for(int i=0;i<marks.size();i++)
	marks[i]=0;
      if (m_capitalization)
	lang->markCapitals(toSend, marks);
    }
  text.setText(toSend);
  for(int i=0;i<marks.size();i++)
    if (marks[i])
      text.mark(i);
  text.setLang(lang);
}

auto_ptr<TextProcessor> createNewTextProcessor(int digitsMode, bool capitalization, bool separation)
{
  return auto_ptr<TextProcessor>(new Processor(digitsMode, capitalization, separation));
}
