/*
	Copyright (c) 2000-2007 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VOICEMAN speech system.

   VOICEMAN speech system is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   VOICEMAN speech system is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
*/

#ifndef __VOICEMAN_TEXT_H__
#define __VOICEMAN_TEXT_H__

#define DEFAULT_OUTPUT_NAME "default"

enum {PARAM_VOLUME = 1, PARAM_PITCH = 2, PARAM_RATE = 3};

class Lang
{
public:
  enum {UPCASE=0, LOWCASE=1, OTHER=2};

  Lang() {}
  virtual ~Lang() {}

  virtual int getCharType(wchar_t c) const = 0;
  virtual bool equalChars(wchar_t c1, wchar_t c2) const = 0;
  virtual void expandNumbers(std::wstring& str, bool singleDigits) const = 0;
  virtual std::wstring getAllChars() const = 0;
  virtual wchar_t toUpper(wchar_t ch) const = 0;
  virtual wchar_t toLower(wchar_t ch) const = 0;
  virtual std::wstring toUpper(const std::wstring& str) const = 0;
  virtual std::wstring toLower(const std::wstring& str) const = 0;
  virtual std::wstring separate(const std::wstring& text) const = 0;
  virtual void markCapitals(const std::wstring& text, std::vector<bool>& marks) const = 0;

};//class Lang;

class TextParam
{
public:
  TextParam() : m_value(50) {}

  TextParam(int value)
  {
    if (value < 0)
      {
	m_value=0;
	return;
      }
    if (value > 100)
      {
	m_value=100;
	return;
      }
    m_value=value;
  }

  void reset()
  {
    m_value=50;
  }

  int getValue() const
  {
    return m_value;
  }

  float getValue(float min, float max) const
  {
    assert(m_value >= 0 && m_value <= 100);
    if (m_value < 0)
      return min;
    if (m_value > 100)
      return max;
    float d=max-min;
    return min+(d*((float)m_value/100));
  }

  const TextParam& operator +=(int step)
  {
    if (step < 0)
      return *this;
    m_value+=step;
    if (m_value>100)
      m_value=100;
    return *this;
  }

  const TextParam& operator -=(int step)
  {
    if (step < 0)
      return *this;
    m_value-=step;
    if (m_value < 0)
      m_value = 0;
    return *this;
  }

  const TextParam& operator =(int value)
  {
    if (value < 0)
      {
      m_value = 0;
      return *this;
      }
    if (value > 100)
      {
	m_value=100;
	return *this;
      }
    m_value=value;
    return *this;
  }

private:
  int m_value;
}; //class TextParam;

class TextItem
{
public:
 TextItem(): m_lang(NULL) {}
  TextItem(const std::wstring& text): m_text(text), m_lang(NULL) {}
  TextItem(const std::string& outputName, const std::wstring& text): m_outputName(outputName), m_text(text), m_lang(NULL) {}
  TextItem(const std::wstring& text, TextParam volume, TextParam pitch, TextParam rate): m_text(text), m_volume(volume), m_pitch(pitch), m_rate(rate), m_lang(NULL) {}
  TextItem(const std::string& outputName, const std::wstring& text, TextParam volume, TextParam pitch, TextParam rate): m_outputName(outputName), m_text(text), m_volume(volume), m_pitch(pitch), m_rate(rate), m_lang(NULL) {}

  std::wstring getText() const
  {
    return m_text;
  }

  void setText(const std::wstring& text)
  {
    m_text=text;
    m_marks.clear();
  }

  void resetParams()
  {
    m_volume.reset();
    m_pitch.reset();
    m_rate.reset();
  }

  void mark(int index)
  {
    assert(index < m_text.length());
    ensureMarksSize(index);
    m_marks[index]=1;
  }

  void unmark(int index)
  {
    assert(index < m_text.length());
    ensureMarksSize(index);
    m_marks[index]=0;
  }

  bool isMarked(int index) const
  {
    assert(index < m_text.length());
    if (index >= m_marks.size())
      return 0;
    return m_marks[index];
  }

  TextParam getVolume() const
  {
    return m_volume;
  }

  void setVolume(TextParam volume)
  {
    m_volume=volume;
  }

  TextParam getPitch() const
  {
    return m_pitch;
  }

  void setPitch(TextParam pitch)
  {
    m_pitch=pitch;
  }

  TextParam getRate() const
  {
    return m_rate;
  }

  void setRate(TextParam rate)
  {
    m_rate=rate;
  }

  std::string getOutputName() const
  {
    return m_outputName;
  }

  void setOutputName(const std::string& outputName)
  {
    m_outputName=outputName;
  }

  const Lang* getLang() const
  {
    return m_lang;
  }

  void setLang(const Lang* lang)
  {
    m_lang=lang;
  }

private:
  void ensureMarksSize(int index)
  {
    if (index < m_marks.size())
      return;
    int oldSize=m_marks.size();
    m_marks.resize(index+1);
    for(int i=oldSize;i<m_marks.size();i++)
      m_marks[i]=0;
  }

private:
  std::wstring m_text;
  std::string m_outputName;
  std::vector<bool> m_marks;
  TextParam m_volume, m_rate, m_pitch;
  const Lang* m_lang;
}; //class TextItem;

enum {DIGITS_NORMAL=0, DIGITS_NONE=1, DIGITS_SINGLE=2};

class TextProcessor
{
public:
  TextProcessor() {}
  virtual ~TextProcessor() {}

  virtual void setDefaultOutput(const std::string& outputName) = 0;
  virtual void setLang(const std::string& outputName, const Lang* lang) = 0;
  virtual void associate(const std::wstring& str, const std::string& outputName) = 0;
  virtual void addReplacement(const std::string& outputName, const std::wstring& oldValue, const std::wstring& newValue) = 0;
  virtual void setSpecialValueFor(wchar_t c, const std::wstring& value) = 0;
  virtual void process(const TextItem& text, std::list<TextItem>& items) const = 0;
  virtual void processLetter(wchar_t c, TextParam volume, TextParam pitch, TextParam rate, std::list<TextItem>& items)=0;
}; //class TextProcessor;

auto_ptr<TextProcessor> createNewTextProcessor(int digitsMode, bool capitalization, bool separation);

#endif //__VOICEMAN_TEXT_H__
