/*
	Copyright (c) 2000-2013 Michael Pozhidaev<msp@altlinux.org>
   This file is part of the VoiceMan speech service.

   VoiceMan speech service is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.

   VoiceMan speech service is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
*/

#ifndef __VOICEMAN_SYSTEM_EXCEPTION_H__
#define __VOICEMAN_SYSTEM_EXCEPTION_H__

#include"logging.h"
#include"VoicemanException.h"

/**\brief The exception class to handle system call errors
 *
 * This exception class is designed for error handling based on analyzing
 * value of UNIX errno variable. The corresponding value is set to this
 * variable in case of errors in all system calls. Also this class has
 * one single-line description used to provide name of system call caused
 * the problem or something else like file name in case of file I/O
 * operations.
 */
class SystemException: public VoicemanException
{
public:
  /**\brief The default constructor
   *
   * This constructor gets error code automatically 
   * by using errno variable and leavs comment string empty.
   */
  SystemException()
    : m_code(errno) {}

  /**\brief The constructor with comment specification
   *
   * This constructor takes comment value as string argument and 
   * gets error code by using errno variable.
   *
   * \param [in] comment The comment value
   */
  SystemException(const std::string& comment)
    : m_code(errno), m_comment(comment) {}

  /**\brief The constructor with error code specification
   *
   * \param [in] code The error code value
   */
  SystemException(int code)
    : m_code(code) {}

  /**\brief The constructor with comment and error code specification
   *
   * \param [in] code The error code value
   * \param [in] comment The comment value
   */
  SystemException(int code, const std::string& comment)
    : m_code(code), m_comment(comment) {}

  /**\brief The destructor*/
  virtual ~SystemException() {}

  /**\brief Returns the error code
   *
   * The error code returned by this method usually the errno value of the
   * corresponding system call error but in general case it can be anything
   * else, because user can set any desired value in exception constructor.
   *
   * \return The code of corresponding error
   */
  int getCode() const
  {
    return m_code;
  }

  /**\brief Returns the string description of the error generated by operating system by error code
   *
   * This method uses strerror() function to get error description provided
   * by operating system itself. This string must not be confused with the
   * value returned by getComment() method which is just single-line
   * comment specified in exception object construction.
   *
   * \return The error description provided by operating system
   */
  std::string getDescr() const
  {
    return strerror(m_code);
  }

  /**\brief returns the comment value
   *
   * This method just returned the single-line comment specified at this exception object creation.
   *
   * \return The single-line error comment
   */
  std::string getComment() const
  {
    return m_comment;
  }

  /**\brief Generates error message by comment and error description
   *
   * This method generates single-line error description which user can
   * print to console or write to log. This description aims to have
   * complete information about error occurred. Usually it includes both
   * descriptions from operating system and specified at exception object
   * creation.
   *
   * \return The complete information about an error
   */
  std::string getMessage() const
  {
    return getComment()+":"+getDescr();
  }

  /**\brief Makes report to logging system
   *
   * This function writes to log a report about error object. It may be not
   * exactly the same value as produced by getMessage() method. The level
   * of message must be specified, because the same error can have
   * different levels in different situations. 
   *
   * \param level The level of error log report
   */
  void makeLogReport(int level) const
  {
    logMsg(level, "System call error:%s", getMessage().c_str());
  }

private:
  int m_code;
  std::string m_comment;
}; //class SystemException;

#define VM_SYS(expr, msg) if (!(expr)) throw SystemException(msg)

#endif //__VOICEMAN_SYSTEM_EXCEPTION_H__
