
#ifndef __VOICEMAN_CONFIG_FILE_EXCEPTION_H__
#define __VOICEMAN_CONFIG_FILE_EXCEPTION_H__

#include"system/logging.h"
#include"VoicemanException.h"

/**\brief The exception object to indicate the configuration file reading error*/
class ConfigFileException: public VoicemanException
{
public:
  /**\brief Codes of various errors during configuration file reading process*/
  enum {
    /**\brief Section header line has invalid format*/
    INVALID_SECTION_HEADER = 1,

    /**\brief Section name omitted*/
    EMPTY_SECTION_NAME = 2,

    /**\brief Parameter placed before first section beginning*/
    PARAMETER_WITHOUT_SECTION = 3,

    /**\brief Value contains illegal characters and is not quoted*/
    INVALID_UNQUOTED_VALUE = 4,

    /**\brief Configuration file line has an invalid beginning*/
    INVALID_LINE_BEGINNING = 5,

    /**\brief Parameter name contains invalid characters*/
    INVALID_PARAMETER_NAME = 6,

    /**\brief There is an invalid character after equal sign*/
    INVALID_CHAR_AFTER_EQUALS = 7,

    /**\brief Line has an invalid ending or incomplete*/
    UNEXPECTED_LINE_END = 8
  };

  /**\brief The constructor only with error code specification
   *
   * \param [in] code The error code for new exception object
   */
  ConfigFileException(int code)
    : m_code(code), m_line(0) {}

  /**\brief The constructor with error code, file name and line number specification
   *
   * \param [in] code The error code for new exception object
   * \param [in] fileName Name of a file with error
   * \param [in] line The line number with error (must not be zero)
   */
  ConfigFileException(int code, const std::string& fileName, size_t line)
    : m_code(code), m_fileName(fileName), m_line(line) 
  {
    assert(m_line > 0);
  }

  /**\breif The destructor*/
  virtual ~ConfigFileException() {}

  /**\brief Returns error code*/
  int getCode() const 
  {
    return m_code;
  }

  /**\brief Returns line number
   *
   * This method has assert() to check current object 
   * was created through the constructor with line number specification.
   */
  size_t getLine() const
  {
    assert(m_line > 0);
    return m_line;
  }

  /**\brief Returns name of a file with error*/
  std::string getFileName() const
  {
    return m_fileName;
  }

  /**\brief Returns the string with error description*/
  std::string getDescr() const
  {
    switch(m_code)
      {
      case INVALID_SECTION_HEADER:
	return "invalid section header format";
      case EMPTY_SECTION_NAME:
	return "missed section name";
      case PARAMETER_WITHOUT_SECTION:
	return "config file section is not opened";
      case INVALID_UNQUOTED_VALUE:
	return "invalid character in value without quotas";
      case INVALID_LINE_BEGINNING:
	return "invalid character before parameter name";
      case INVALID_PARAMETER_NAME:
	return "invalid character in parameter name";
      case INVALID_CHAR_AFTER_EQUALS:
	return "invalid character after \'=\' sign";
      case UNEXPECTED_LINE_END:
	return "unexpected line end or line is incomplete";
      default:
	assert(0);
      } //switch(m_code);
    return "";
  }

  /**\brief Prepares complete error description with file name and line number if there are any*/
  std::string getMessage() const 
  {
    if (m_line == 0)
      return getDescr();
    std::ostringstream s;
    s << m_fileName << "(" << m_line << "):" << getDescr();
    return s.str();
  }

  /**\brief Sends error description to logging system*/
  void makeLogReport(int level) const
  {
    logMsg(level, "config file error:%s", getMessage().c_str());
  }

private:
  const int m_code;
  const std::string m_fileName;
  const size_t m_line;
}; // class ConfigFileException;

#endif //__VOICEMAN_CONFIG_FILE_EXCEPTION_H__
